
/*
 * Copyright 2009  Bernd Buschinski b.buschinski@web.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License or (at your option) version 3 or any later version
 * accepted by the membership of KDE e.V. (or its successor approved
 * by the membership of KDE e.V.), which shall act as a proxy
 * defined in Section 14 of version 3 of the license.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "bkodama.h"

#include "brand.h"

#include <KStandardDirs>
#include <KConfigDialog>
#include <KWindowSystem>
#include <KApplication>

#include <QTimer>
#include <QPainter>
#include <QSvgRenderer>
#include <QDesktopWidget>
#include <QGraphicsSceneMouseEvent>

static const QStringList cWalkOutList = QStringList() << "walk00-test.svg" << "walk01-test.svg" << "walk02-test.svg" << "walk03-test.svg" <<  "walk04-test.svg" << "walk05-test.svg" << "walk06-test.svg";
static const int cWalkOutTime = 800; //ms

static const QStringList cFadeInList = QStringList() << "sitting01-test.svg";
static const int cFadeInTime = 800; //ms

static const QStringList cFadeOutSittingList = QStringList() << "sitting01-test.svg";
static const int cFadeOutSittingTime = 800; //ms

static const QStringList cFadeOutStandingList = QStringList() << "stand00-test.svg";
static const int cFadeOutStandingTime = 800; //ms

static const QStringList cStandUpList = QStringList() << "sitting01-test.svg" << "stand03-test.svg" << "stand02-test.svg" << "stand01-test.svg" << "stand01-test.svg" << "stand00-test.svg";
static const int cStandUpTime = 720; //ms

static const QStringList cHeadSpinList = QStringList() << "spin00-test.svg" << "spin00-test.svg" << "spin01-test.svg" << "spin01-test.svg" << "spin02-test.svg" << "spin02-test.svg" << "spin03-test.svg" << "spin04-test.svg" << "spin03-test.svg" << "spin05-test.svg" << "spin02-test.svg"  << "sitting01-test.svg";
static const int cHeadSpinTime = 1800; //ms

//msecs
static const int cTimerFast = 30;
static const int cShortBreak = 2000;
static const int cLongBreak = 5000;
static const int cVeryLongBreak = 35000;

using namespace Plasma;

bkodamaapplet::bkodamaapplet(QObject * parent, const QVariantList & args)
    : Plasma::Applet(parent, args)
{
    m_sound = NULL;
    m_audioOutput = NULL;
    m_sound_enabled = true;
    m_sound_url = KGlobal::dirs()->findAllResources("data", "bkodama/head-spin3.ogg").first();
    m_sound_volume = 100;
    m_alpha = 100;
    m_alpha_modifier = 0;
    m_X_modifier = 0;
    m_ImageIndex = 0;
    m_mouse_pressed = false;

    m_animation_pos = 0;

    m_timer = new QTimer (this);
    connect (m_timer, SIGNAL (timeout ()), this, SLOT (animation() ));

    loadSVGList();
    initStandUp();
    setImage(cStandUpList.first());

    setHasConfigurationInterface (true);

    setBackgroundHints(NoBackground);
    setMinimumSize(QSizeF(18.75f, 30.0f));
    setMaximumSize(QSizeF(156.25f,250.0f));
    setAspectRatioMode( Plasma::KeepAspectRatio );
}

bkodamaapplet::~bkodamaapplet()
{
    while (!m_SVGRenderList.isEmpty())
    {
        delete m_SVGRenderList.takeFirst();
    }

    if (m_timer)
    {
        disconnect(m_timer, 0,0,0);
        if (m_timer->isActive())
            m_timer->stop();
        delete m_timer;
    }

    delete m_audioOutput;
    delete m_sound;
}

void bkodamaapplet::init()
{
    readConfiguration();

    initSound();

    //kDebug() << KApplication::desktop()->numScreens();
    //kDebug() << KApplication::desktop()->isVirtualDesktop();
    if (KApplication::desktop()->numScreens() > 1 || KApplication::desktop()->isVirtualDesktop())
    {
        //int tScreen = KApplication::desktop()->screenNumber( view()->parentWidget() ); //always screen 0
        int tScreen = KApplication::desktop()->screenNumber( QCursor::pos() ); //unreliable
        //kDebug() << "tScreen" << tScreen;

        m_screen = KWindowSystem::workArea().intersect(KApplication::desktop()->screenGeometry(tScreen));
        //m_screen = KApplication::desktop()->screen()->geometry();
    }
    else
    {
        m_screen = KWindowSystem::workArea();
    }
    //m_screen = screenRect();

    kDebug() << "width: " << m_screen.width();
    kDebug() << "height: " << m_screen.height();

    // 500/800 -> svg size
    setMaximumSize( QSizeF(m_screen.height()*500.0f/800.0f/5.0f, m_screen.height()/5.0) );
    resize( QSizeF( maximumWidth(), maximumHeight()) );

    updateScaledImage();

    m_timer->start ( brand( cLongBreak ) + cShortBreak );
}

void bkodamaapplet::animation()
{
    if (m_mouse_pressed)
    {
        return;
    }

    bool animation_ended = false;
    setTimerSpeed( cTimerFast );

    switch(m_animation)
    {
    case FadeOutWalk:
        {
            m_alpha += m_alpha_modifier;
            ++m_animation_pos;
            setAlphaToPercent(m_alpha);
            if ((m_animation_pos * cTimerFast) >= cWalkOutTime)
            {
                animation_ended = true;
                setAlphaToPercent(m_alpha);
                break;
            }
            int tImage = int((m_animation_pos * (float)cTimerFast * (float)cWalkOutList.size()) / (float)cWalkOutTime );
            setImage( cWalkOutList.at(tImage) );
            setAlphaToPercent(m_alpha);
            QRectF tGeometry = geometry();
            setPos(tGeometry.x() - m_X_modifier, tGeometry.y());
        }
        break;
    case FadeOutSitting:
        if (m_alpha == 100)
        {
            setImage( cFadeOutSittingList.first() );
        }
        m_alpha += m_alpha_modifier;
        setAlphaToPercent(m_alpha);
        if(m_alpha <= 0.0f)
        {
            animation_ended = true;
        }
        break;
    case FadeOutStanding:
        if (m_alpha == 100)
        {
            setImage( cFadeOutStandingList.first() );
        }
        m_alpha += m_alpha_modifier;
        setAlphaToPercent(m_alpha);
        if(m_alpha <= 0.0f)
        {
            animation_ended = true;
        }
        break;
    case FadeIn:
        if (m_alpha == 0)
        {
            setImage( cFadeInList.first() );
        }
        m_alpha += m_alpha_modifier;
        setAlphaToPercent(m_alpha);
        if(m_alpha >= 100.0f)
        {
            animation_ended = true;
        }
        break;
    case StandUp:
        {
            ++m_animation_pos;
            if ((m_animation_pos * cTimerFast) >= cStandUpTime)
            {
                animation_ended = true;
                break;
            }
            int tImage = int( (m_animation_pos * (float)cTimerFast * (float)cStandUpList.size()) / (float)cStandUpTime);
            setImage( cStandUpList.at(tImage) );
        }
        break;
    case HeadSpin:
        {
            if (m_sound_enabled && m_animation_pos == 0)
            {
                kDebug() << "play";
                m_sound->play();
            }
            ++m_animation_pos;
            if ((m_animation_pos * cTimerFast) >= cHeadSpinTime)
            {
                animation_ended = true;
                break;
            }
            int tImage = int( (m_animation_pos * (float)cTimerFast * (float)cHeadSpinList.size()) / (float)cHeadSpinTime );
            setImage( cHeadSpinList.at(tImage) );
        }
        break;
    default:
        kDebug() << "unknown1 animation " << m_animation;
        break;
    }

    if (animation_ended)
    {
        m_animation_pos = 0;
        m_X_modifier = 0;
        m_alpha_modifier = 0;

        int tRandom;

        switch (m_animation)
        {
        case FadeOutWalk:
            initFadeIn();
            break;
        case FadeIn:
            tRandom = brand(100);
            if (tRandom > 96)
            {
                initFadeOutSitting();
            }
            else if (tRandom > 20)
            {
                initHeadSpin();
            }
            else
            {
                initStandUp();
            }
            break;
        case HeadSpin:
            tRandom = brand(100);
            if (tRandom > 95)
            {
                initHeadSpin();
            }
            else
            {
                initStandUp();
            }
            break;
        case StandUp:
            tRandom = brand(100);
            if (tRandom > 95 || walkOutDistance() > geometry().x())
            {
                initFadeOutStanding();
            }
            else
            {
                initFadeOutWalk();
            }
            break;
        case FadeOutSitting:
        case FadeOutStanding:
            initFadeIn();
            break;

        default:
            kDebug() << "unknown2 animation " << m_animation;
            break;
        }
    }

    update();
}

void bkodamaapplet::setAlphaToPercent(float aAlphaPercent)
{
    //QTime tTime = QTime::currentTime();
    if (aAlphaPercent >= 100.0f)
    {
        m_alpha_img = m_scaled_img;
        return;
    }

    if (aAlphaPercent <= 0.0f)
    {
        m_alpha_img.fill(Qt::transparent);
        return;
    }

    int tBytePerLine = m_alpha_img.bytesPerLine();
    int tHeight = m_alpha_img.height();
    for (int i = 0; i < tHeight; ++i)
    {
        QRgb* tRgb = (QRgb*)m_scaled_img.scanLine(i);
        QRgb* tLineAlpha = (QRgb*)m_alpha_img.scanLine(i);
        for (int j = 0; j < tBytePerLine; j += sizeof(QRgb))
        {
            int tAlpha = qAlpha(*tRgb);
            if (tAlpha > 0)
            {
                *tLineAlpha = qRgba(qRed(*tRgb),qGreen(*tRgb), qBlue(*tRgb), ((float)tAlpha/100.0f) * aAlphaPercent);
                //kDebug() << "alpha: " << ((float)qAlpha(*tRgb)/100.0) * aAlphaPercent;
            }
            ++tRgb;
            ++tLineAlpha;
        }
    }
    //kDebug() << tTime.elapsed();
}

void bkodamaapplet::constraintsEvent (Plasma::Constraints constraints)
{
    if (constraints & Plasma::FormFactorConstraint)
    {
        setBackgroundHints(NoBackground);
    }

    if ( constraints & Plasma::SizeConstraint )
    {
        updateScaledImage();
        setAlphaToPercent(m_alpha);
        update();
    }
}

void bkodamaapplet::updateScaledImage()
{
    //QTime tTime = QTime::currentTime();
    QRectF tRect = geometry();
    m_scaled_img = QImage ( tRect.width(), tRect.height() , QImage::Format_ARGB32 );
    m_scaled_img.fill( Qt::transparent );
    m_SVGRenderList[m_ImageIndex]->resize(tRect.width(), tRect.height());
    QPainter tPaint;
    tPaint.begin( &m_scaled_img );
    m_SVGRenderList[m_ImageIndex]->paint(&tPaint, m_scaled_img.rect());
    tPaint.end();
    //kDebug() << tTime.elapsed();

    m_alpha_img = m_scaled_img;
}

void bkodamaapplet::paintInterface (QPainter * p, const QStyleOptionGraphicsItem * option, const QRect & contentsRect)
{
    Q_UNUSED (option);
    Q_UNUSED (contentsRect);

    p->setRenderHint (QPainter::SmoothPixmapTransform);
    p->setRenderHint (QPainter::Antialiasing);

    p->drawImage(0,0,m_alpha_img);
}

void bkodamaapplet::setTimerSpeed(int aMSecs)
{
    if (m_timer->interval() != aMSecs)
    {
        m_timer->start( aMSecs );
    }
}

void bkodamaapplet::loadSVGList()
{
    QStringList tList = KGlobal::dirs()->findAllResources("data", "bkodama/*.svg");

    foreach(const QString& tFile, tList)
    {
        kDebug() << tFile;
        Svg* tRender = new Svg();
        tRender->setImagePath( tFile );
        if (tRender->isValid())
        {
            m_SVGRenderList.append(tRender);
        }
        else
        {
            kDebug() << "invalid svg: " << tFile;
            delete tRender;
            continue;
        }
    }
}

//searches for the aImageName in the list of svgs and sets m_ImageIndex if found
bool bkodamaapplet::setImage(const QString& aImageName)
{
    for (int i = 0; i < m_SVGRenderList.size(); ++i)
    {
        if (m_SVGRenderList.at(i)->imagePath().endsWith(aImageName))
        {
            //nothing changed
            if (i == m_ImageIndex)
            {
                return true;
            }
            m_ImageIndex = i;
            updateScaledImage();
            return true;
        }
    }
    kDebug() << "could not find imageindex for " << aImageName;
    return false;
}

void bkodamaapplet::initFadeOutWalk()
{
    m_animation = FadeOutWalk;
    m_alpha = 100;
    m_alpha_modifier = -(float(cTimerFast) * 100.0f) / (float)cWalkOutTime;
    m_X_modifier = int( ((float)cTimerFast * (219.48f/500.0f) * (float)geometry().width()) / (float)cWalkOutTime + 0.5f);
    setTimerSpeed( brand( cShortBreak ) + cShortBreak );
}

float bkodamaapplet::walkOutDistance()
{
//100/(100/(800/30)) * ((219.48/500)*100)/(800/30)

//original stuff:
//m_alpha_modifier = (100.0f / ( (float)cWalkOutTime / cTimerFast ));
//m_X_modifier = int((float((219.48f/500.0f)*m_position.width())/float((float)cWalkOutTime / (float)cTimerFast ))+0.5);

// yes this is ugly
// m_X_modifier = distance per tick
// m_alpha_modifier = 100/m_alpha_modifier how many ticks
//     float tAlphaTicks = (float)cWalkOutTime / float(cTimerFast);
//     float tXModifier = ((float)cTimerFast * (219.48f/500.0f) * m_position.width()) / (float)cWalkOutTime;
    //return tAlphaTicks * tXModifier + m_position.width()/8.0f + 0.5f;

//(a/b) * (b*d*e)/a + e/8
//d * e + e / 8
//d * e + e * 0.125
// d = (219.48f/500.0f)
// d = 0.43896f
//mathimized
    float tWidth = geometry().width();
    return 0.43896f * tWidth + tWidth * 0.125f + 0.5f;
}

void bkodamaapplet::initFadeOutSitting()
{
    m_animation = FadeOutSitting;
    m_alpha = 100;
    m_alpha_modifier = -(float)cTimerFast * 100.0f / (float)cFadeOutSittingTime;
    setTimerSpeed( brand( cLongBreak ) + cShortBreak );
}

void bkodamaapplet::initFadeOutStanding()
{
    m_animation = FadeOutStanding;
    m_alpha = 100;
    m_alpha_modifier = -(float)cTimerFast * 100.0f / (float)cFadeOutStandingTime ;
    setTimerSpeed( brand( cLongBreak ) + cShortBreak );
}

void bkodamaapplet::initFadeIn()
{
    m_animation = FadeIn;
    m_alpha_modifier = (float)cTimerFast * 100.0 / (float)cFadeInTime;
    m_alpha = 0;
    setTimerSpeed( brand( cVeryLongBreak ) + cLongBreak );
    //setTimerSpeed( brand( 1 ) + cLongBreak );

    unsigned int x = brand( m_screen.width() - geometry().width());
    unsigned int y = brand( m_screen.height() - geometry().height());

    setPos(x,y);
//     kDebug() << "x: " << x;
//     kDebug() << "y: " << y;
//     kDebug() << "screen width: " << m_screen.width();
//     kDebug() << "screen height: " << m_screen.height();
}

void bkodamaapplet::initHeadSpin()
{
    m_animation = HeadSpin;
    m_alpha = 100;
    setTimerSpeed( brand( cLongBreak ) + cShortBreak );
}

void bkodamaapplet::initStandUp()
{
    m_animation = StandUp;
    m_alpha = 100;
    setTimerSpeed( brand( cLongBreak ) + cShortBreak );
}

void bkodamaapplet::createConfigurationInterface(KConfigDialog *aParent)
{
    QWidget *widget = new QWidget;
    ui.setupUi (widget);

    connect(aParent, SIGNAL(accepted()), this, SLOT(configAccepted()));

    //Sound
    ui.soundEnabled->setChecked (m_sound_enabled);
    ui.soundVolumeLabel->setEnabled (m_sound_enabled);
    ui.soundVolume->setEnabled (m_sound_enabled);
    ui.soundVolume->setSliderPosition (m_sound_volume);

    aParent->addPage(widget, i18n("General"), icon());
}

void bkodamaapplet::configAccepted()
{
    KConfigGroup cg = config();

    //Sound
    m_sound_enabled = (ui.soundEnabled->checkState () == Qt::Checked);
    cg.writeEntry ("SoundEnabled", m_sound_enabled);
    if ( m_sound_enabled )
    {
        m_sound_volume = ui.soundVolume->value ();
        cg.writeEntry ("SoundVolume", m_sound_volume);
        initSound();
    }

    //mouse - undo the mouse clicked
    m_mouse_pressed = false;

    update();
}

void bkodamaapplet::mouseReleaseEvent (QGraphicsSceneMouseEvent * event)
{
    m_mouse_pressed = false;
    event->accept();
}

void bkodamaapplet::mousePressEvent (QGraphicsSceneMouseEvent * event)
{
    m_mouse_pressed = true;
    event->accept();
}

void bkodamaapplet::resizeEvent(QGraphicsSceneResizeEvent * event)
{
//it crashes if the don't stop the sound here, I wonder why
    if (m_sound && (m_sound->state() == Phonon::PlayingState || m_sound->state() == Phonon::BufferingState))
    {
        m_sound->stop();
    }
    Plasma::Applet::resizeEvent(event);
}

QVariant bkodamaapplet::itemChange(GraphicsItemChange change, const QVariant & value)
{
    //bypass Plasma::Applet when it comes to position changes, because otherwise
    //an locked applet can not move
    QVariant ret;
    if (change == QGraphicsItem::ItemPositionChange || change == QGraphicsItem::ItemPositionHasChanged)
    {
        ret = QGraphicsWidget::itemChange(change, value);
    }
    else
    {
        ret = Plasma::Applet::itemChange(change, value);
    }
    return ret;
}

void bkodamaapplet::readConfiguration()
{
    KConfigGroup cg = config();

    //Sound
    m_sound_enabled = cg.readEntry ("SoundEnabled", true);
    m_sound_volume = cg.readEntry ("SoundVolume", 100);
}

void bkodamaapplet::initSound()
{
    if (m_sound_enabled)
    {
        if (!m_sound)
        {
            m_sound = new Phonon::MediaObject (this);
            m_audioOutput = new Phonon::AudioOutput (Phonon::MusicCategory, this);
            m_sound->setCurrentSource (m_sound_url);
            createPath(m_sound, m_audioOutput);
        }
        m_audioOutput->setVolume (m_sound_volume);
    }
}

#include "bkodama.moc"
