/*
 * Copyright 2013 Canonical Ltd.
 *
 * This file is part of powerd.
 *
 * powerd is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * powerd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <string.h>
#include <uuid/uuid.h>
#include <glib.h>

#include "powerd-internal.h"

struct sync_work_data {
    GCond cond;
    GMutex mutex;
    gboolean done;
    int (*func)(gpointer);
    gpointer data;
    int return_val;
};

static gboolean run_mainloop_sync_worker(gpointer data)
{
    struct sync_work_data *work_data = data;

    g_mutex_lock(&work_data->mutex);
    work_data->return_val = work_data->func(work_data->data);
    work_data->done = TRUE;
    g_cond_signal(&work_data->cond);
    g_mutex_unlock(&work_data->mutex);

    return FALSE;
}

static int __powerd_run_mainloop_sync(int (*func)(gpointer), gpointer data)
{
    struct sync_work_data work_data;

    g_cond_init(&work_data.cond);
    g_mutex_init(&work_data.mutex);
    work_data.done = FALSE;
    work_data.func = func;
    work_data.data = data;

    g_mutex_lock(&work_data.mutex);
    g_timeout_add(0, run_mainloop_sync_worker, &work_data);
    while (!work_data.done)
        g_cond_wait(&work_data.cond, &work_data.mutex);
    g_mutex_unlock(&work_data.mutex);

    g_mutex_clear(&work_data.mutex);
    g_cond_clear(&work_data.cond);

    return work_data.return_val;
}

/*
 * Run the supplied function on the main loop, passing it the
 * supplied data, and wait for it to complete. If currently
 * executing on the main loop then the function is simply called
 * inline.
 */
int powerd_run_mainloop_sync(int (*func)(gpointer), gpointer data)
{
    if (powerd_is_mainloop())
        return func(data);
    return __powerd_run_mainloop_sync(func, data);
}

/*
 * Calculate a hash from a UUID. For use with GHashTables.
 */
guint powerd_uuid_hash(gconstpointer key)
{
    guint *data = (guint *)key;
    guint hash, i;

    /*
     * Treat the UUID as an array of guints and do a simple xor
     * hash. uuid_t is an even multiple of the size of guint for
     * 32- or 64-bit integers, but even if it weren't this
     * implementation would just leave off the trailing bytes and
     * still produce a reasonable hash.
     */
    hash = data[0];
    for (i = 1; i < sizeof(uuid_t) / sizeof(hash); i++)
        hash ^= data[i];

    return hash;
}

/*
 * Compare two UUIDs and return TRUE if equal and FALSE
 * otherwise. For use with GHashTables.
 */
gboolean powerd_uuid_equal(gconstpointer a, gconstpointer b)
{
    return !memcmp(a, b, sizeof(uuid_t));
}
