/**
 * \file   mlinfo.cpp
 * \author The GenABEL team
 *
 * \brief Contains the class implementation of the mlinfo class.
 */
/*
 *
 * Copyright (C) 2009--2016 Various members of the GenABEL team. See
 * the SVN commit logs for more details.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 */


#include <sstream>
#include <fstream>
#include <iostream>
#include "mlinfo.h"


/**
 * \brief Constructor that reads SNP information from an
 * <tt>.mlinfo</tt> or <tt>.info</tt> file generated by the imputation
 * software.
 *
 * \param filename Name of the mlinfo file
 * \param mapname Name of the map file
 * \param flipMAF Indicates whether alleles should be flipped based on
 * MAF (\sa cmdvars::getFlipMAF and cmdvars::flipMAF)
 */
mlinfo::mlinfo(const char * filename,
               const char * mapname,
               const bool flipMAF)
{
    char tmp[1048576];
    unsigned int nlin = 0;
    std::ifstream infile(filename);
    if (infile.is_open())
    {
        while (infile.good())
        {
            infile >> tmp;
            nlin++;
        }
        nlin--; // Subtract one, the previous loop added 1 too much
    } else {
        std::cerr << "mlinfo: cannot open info file "
                  << filename << std::endl;
        exit(1);
    }
    infile.close();

    if (nlin % 7)
    {
        std::cerr << "mlinfo: number of columns != 7 in "
                  << filename << std::endl;
        exit(1);
    }
    nsnps = static_cast<int>((nlin / 7) - 1);
    std::cout << "Number of SNPs = " << nsnps << std::endl;
    name    = new std::string[nsnps];
    A1      = new std::string[nsnps];
    A2      = new std::string[nsnps];
    Freq1   = new double[nsnps];
    MAF     = new double[nsnps];
    Quality = new double[nsnps];
    Rsq     = new double[nsnps];
    map     = new std::string[nsnps];
    if (flipMAF)
    {
        allelesFlipped = std::vector<bool>(nsnps, false);
    }

    infile.open(filename);
    if (!infile)
    { // file couldn't be opened
        std::cerr << "mlinfo: cannot open info file "
                  << filename << std::endl;
        exit(1);
    }
    /* Read the header and discard it */
    for (int i = 0; i < 7; i++)
        infile >> tmp;

    for (int i = 0; i < nsnps; i++)
    {
        infile >> tmp;
        name[i] = tmp;
        infile >> tmp;
        A1[i] = tmp;
        infile >> tmp;
        A2[i] = tmp;
        infile >> tmp;
        Freq1[i] = atof(tmp);
        infile >> tmp;
        MAF[i] = atof(tmp);
        infile >> tmp;
        Quality[i] = atof(tmp);
        infile >> tmp;
        Rsq[i] = atof(tmp);
        map[i] = "-999";
    }
    infile.close();

    if (mapname != NULL)
    {
        std::ifstream instr(mapname);
        int BFS = 1048576;
        char *line = new char[BFS];
        char *tmp  = new char[BFS];

        if (!instr.is_open())
        {
            std::cerr << "mlinfo: cannot open map file "
                      << mapname << std::endl;
            exit(1);
        }

        instr.getline(line, BFS);

        for (int i = 0; i < nsnps; i++)
        {
            instr.getline(line, BFS);
            std::stringstream line_stream(line);
            line_stream >> tmp >> map[i];
        }

        instr.close();

        delete[] line;
        delete[] tmp;
    }
}

/**
 * \brief Destructor. Delete all allocated arrays.
 */
mlinfo::~mlinfo()
{
    delete[] mlinfo::name;
    delete[] mlinfo::A1;
    delete[] mlinfo::A2;
    delete[] mlinfo::Freq1;
    delete[] mlinfo::MAF;
    delete[] mlinfo::Quality;
    delete[] mlinfo::Rsq;
    delete[] mlinfo::map;
}
