#!/usr/bin/python

"""Fake pwrkap driver that simulates CPUs and a power meter for them."""
# (C) Copyright IBM Corp. 2008-2009
# Licensed under the GPLv2.
import pwrkap_data
import discovery
import random
import datetime
import thread
import threading
import time
import traceback

cpu_domains = None
cpus = None
idomains = None
pm = None
em = None

class fake_energy_meter(pwrkap_data.energy_meter):
	def __init__(self, pm_devices):
		self.pm_devices = pm_devices
		self.power_use = None
		self.energy_use = None
		self.last_update = None
		self.keep_running = True

		for pmdev in self.pm_devices:
			assert pmdev.get_prefix() == "fakecpu"

		# These must run last
		self.update_energy_use()
		thread.start_new_thread(self.run, ())

	def __del__(self):
		self.keep_running = False

	def __setstate__(self, data):
		self.__dict__ = data
		self.energy_use = 0
		thread.start_new_thread(self.run, ())

	def run(self):
		"""Periodically update energy use."""
		self.keep_running = True
		while self.keep_running:
			try:
				self.update_energy_use()
			except Exception, e:
				print e
				traceback.print_exc()
			time.sleep(1)

	def update_energy_use(self):
		"""Update the energy use counter."""
		now = datetime.datetime.utcnow()

		if self.last_update == None:
			self.energy_use = 0
			self.power_use = 0
			self.last_update = now
			return

		td = now - self.last_update
		time_delta =  td.microseconds + \
			     (td.seconds * 1000000) + \
			     (td.days * 86400000000)
		if time_delta == 0:
			time_delta = 1
		time_delta = float(time_delta) / 1000000

		# Baseline this system uses about 150W.
		delta_e = 0 #time_delta * (150 + random.randint(-5, 20))

		# CPU energy costs are partially based on the current frequency
		# and partly on the number of clocks used.  Scales are designed
		# such that a 3GHz CPU consumes 120W at full use.
		for cpu in self.pm_devices:
			nr_clocks = (0.4 + 0.6 * cpu.get_utilization()) * cpu.get_current_power_state()
			nr_clocks = nr_clocks * 0.00004
			delta_e = delta_e + time_delta * nr_clocks

		self.energy_use = self.energy_use + delta_e
		self.power_use = (0.666 * self.power_use) + (0.333 * (delta_e / time_delta))
		self.last_update = now

	def read(self):
		return self.energy_use

	def get_latency(self):
		return 0;

	def inventory(self):
		return ("fakemeter", {})

class fake_power_meter(pwrkap_data.power_meter):
	def __init__(self, energy_meter):
		self.energy_meter = energy_meter

	def read(self):
		return self.energy_meter.power_use

	def get_latency(self):
		return 0;

	def inventory(self):
		return ("fakemeter", {})

def build_pstate_table(min, max, step):
	"""Build a fake power state table."""
	states = []

	for mhz in range(min, max + 1, step):
		states.append((mhz, (1.0 * mhz) / max))

	return states

MIN_FAKE_CPU_SPEED = 2000000
MAX_FAKE_CPU_SPEED = 4000000
FAKE_CPU_SPEED_STEP = 1000000
class fake_cpu(pwrkap_data.device):
	def __init__(self, id):
		global MAX_FAKE_CPU_SPEED
		self.max_pstate = MAX_FAKE_CPU_SPEED
		self.id = id
		self.cps_reads = 1
		self.last_cps = MAX_FAKE_CPU_SPEED
		self.master_cpu = None
		self.last_util_read_time = datetime.datetime.utcnow()
		self.last_util_read = 0.5

		self.pstates = build_pstate_table(
			MIN_FAKE_CPU_SPEED,
			MAX_FAKE_CPU_SPEED,
			FAKE_CPU_SPEED_STEP
		)
		random.seed()

	def get_power_states(self):
		if self.master_cpu != None:
			return self.master_cpu.get_power_states()
		return self.pstates

	def get_max_power_state(self):
		if self.master_cpu != None:
			return self.master_cpu.get_max_power_state()
		return self.max_pstate

	def set_max_power_state(self, max_pstate):
		if self.master_cpu != None:
			return self.master_cpu.set_max_power_state(max_pstate)
		valid_state = False
		for (state, potential) in self.pstates:
			if max_pstate == state:
				valid_state = True
		assert valid_state
		self.max_pstate = max_pstate
		if self.last_cps > self.max_pstate:
			self.last_cps = self.max_pstate

	def find_index_of_pstate(self, pstate):
		for i in range(0, len(self.pstates)):
			if self.pstates[i][0] == pstate:
				return i
		return None

	def get_current_power_state(self):
		if self.master_cpu != None:
			return self.master_cpu.last_cps

		self.cps_reads = self.cps_reads + 1
		if self.cps_reads > 8 or self.last_cps > self.max_pstate:
			self.cps_reads = 0
			max_pstate = self.find_index_of_pstate(self.get_max_power_state())
			pstate = random.randint(0, max_pstate)
			(speed, junk) = self.pstates[pstate]
			self.last_cps = speed
		assert self.last_cps <= self.max_pstate
		return self.last_cps

	def get_utilization(self):
		now = datetime.datetime.utcnow()
		if (now - self.last_util_read_time) < pwrkap_data.ONE_SECOND:
			return self.last_util

		self.last_util = (90.0 + random.uniform(-80, 10)) / 100
		self.last_util_read_time = now
		return self.last_util

	def get_utilization_details(self):
		return {self.get_name(): self.get_utilization()}

	def get_id(self):
		return self.id

	def get_name(self):
		return self.get_prefix() + str(self.id)

	def inventory(self):
		key = self.get_prefix() + str(self.id)
		obj = {"states": self.get_power_states()}

		return (key, obj)

	def get_prefix(self):
		return "fakecpu"

	def start_load(self):
		return False

	def stop_load(self):
		# XXX Implement me
		pass

def fake_cpu_discover_old():
	"""Discover fake system CPUs."""
	global cpu_domains, cpus, idomains

	cpu_domains = []
	cpus = []
	idomains = [[], [], []]

	# Simulate 4 dual-core CPUs
	for cpuid in range(0, 4):
		core0 = fake_cpu(2 * cpuid)
		core1 = fake_cpu(2 * cpuid + 1)
		core1.master_cpu = core0
		domain = pwrkap_data.device_domain([core0, core1])
		cpu_domains.append(domain)
		cpus.append(core0)
		cpus.append(core1)
		idomains[cpuid / 2].append(core0)
		idomains[cpuid / 2].append(core1)

	# And a spurious 9th core for fun
	core0 = fake_cpu(8)
	domain = pwrkap_data.device_domain([core0])
	cpu_domains.append(domain)
	cpus.append(core0)
	idomains[2].append(core0)

def fake_cpu_discover():
	"""Discover fake system CPUs."""
	global cpu_domains, cpus, idomains

	cpu_domains = []
	cpus = []
	idomains = [[]]

	# Simulate a single quad-core clovertown
	core0 = fake_cpu(0)
	core1 = fake_cpu(1)
	core2 = fake_cpu(2)
	core3 = fake_cpu(3)
	core1.master_cpu = core0
	core3.master_cpu = core2
	domain0 = pwrkap_data.device_domain([core0, core1])
	domain1 = pwrkap_data.device_domain([core2, core3])
	cpu_domains.append(domain0)
	cpu_domains.append(domain1)
	cpus = [core0, core1, core2, core3]
	idomains = [[core0, core1, core2, core3]]

def fake_meter_discover():
	"""Discover fake power meters."""
	global cpus, pm, em

	em = fake_energy_meter(cpus)
	pm = fake_power_meter(em)

def fake_system_discover():
	"""Discover fake power domains."""
	global cpu_domains, idomains, pm, em

	for i in range(0, 1):
		fake_cpu_discover()
		fake_meter_discover()
		dm = pwrkap_data.power_domain(cpu_domains, idomains, pm, em, 1000)
		dm.snapshot()
		discovery.PWRKAP_POWER_DOMAINS.append(dm)

def fake_init():
	"""Set up discovery functions."""

	discovery.PWRKAP_POWER_DOMAIN_DISCOVERY.append(fake_system_discover)

	return True

fake_init()
