# -*- coding: utf-8 -*-
# PYLang 0.0.4 http://launchpad.net/pylang
# Copyright (C) 2008-2013 Marcos Alvarez Costales https://launchpad.net/~costales
#
# PYLang is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# PYLang is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with PYLang; if not, see http://www.gnu.org/licenses for more
# information.


from gi.repository import Gtk, Gdk, GObject

import os, webbrowser
from multiprocessing import Process

import gettext
from gettext import gettext as _
gettext.textdomain('pylang')

from about import WinAbout
from utils import Utils


def speak_sentence(lang, sentence):
    os.system("espeak -v" + lang + " -s136 ?" + sentence)
    
    


class Practice:
    TIME_STATUS_BAR = 8
    def __init__(self, sentences):
        self.utils = Utils()
        self.sentences = sentences
        self.translate_lang = {}
        
        self.builder = Gtk.Builder()
        self.builder.set_translation_domain("pylang")
        self.builder.add_from_file('/usr/share/pylang/ui/main.ui')
        
        self._set_objects_name()
        self._set_initial_values()
        
        # Set initial sentences
        self.extra_lives = 3
        if self.sentences.load_file():
            self._set_window_status('new_sentence')
            self._set_new_sentence()
        else:
            self._set_window_status('error_initial_file')
            self.popup_error()
    
        self.builder.connect_signals(self)
        self._set_available_langs()
        Gtk.main()
    
    def _set_objects_name(self):
        self.winMain = self.builder.get_object('main')
        
        self.cb_languages   = self.builder.get_object('cb_languages')
        self.box_subject    = self.builder.get_object('eventbox_subject')
        self.box_verb       = self.builder.get_object('eventbox_verb')
        self.box_verb_tense = self.builder.get_object('eventbox_verb_tense')
        self.box_option     = self.builder.get_object('eventbox_option')
        self.box_object     = self.builder.get_object('eventbox_object')
    
        self.entry_sentence = self.builder.get_object('entry_sentence')
        self.btn_lives      = self.builder.get_object('btn_lives')
        self.btn_try_next   = self.builder.get_object('btn_try_next')
        self.img_state      = self.builder.get_object('img_state')
        self.btn_play       = self.builder.get_object('btn_play')
        self.lbl_statusbar  = self.builder.get_object('lbl_statusbar')
    
        self.lbl_subject = self.builder.get_object('lbl_subject')
        self.lbl_verb    = self.builder.get_object('lbl_verb')
        self.lbl_tense   = self.builder.get_object('lbl_tense')
        self.lbl_option  = self.builder.get_object('lbl_option')
        self.lbl_object  = self.builder.get_object('lbl_object')
        
        self.statusbar = self.builder.get_object('statusbar')
        self.box_donate = self.builder.get_object('boxDonate')
        
    def _set_initial_values(self):
        # Background colors
        self.box_subject.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#c7c8ff'))
        self.box_verb.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#ceffd1'))
        self.box_verb_tense.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#ffc0c0'))
        self.box_option.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#fffbbd'))
        self.box_object.modify_bg(Gtk.StateType.NORMAL, Gdk.color_parse('#ffd4f7'))
        
        if self.utils.get_config_value('hidedonatebtn') == 'Yes':
            self.box_donate.hide()
        
    def _set_available_langs(self):
        i = 0
        previous_lang = self.utils.get_config_value('initial_language')
        if not previous_lang:
            previous_lang = 'english'
            
        for lang in self.sentences.get_all_langs():
            lang_in_cb = _(lang.title().replace('_', ' '))  
            self.translate_lang[lang_in_cb] = lang.lower()
            self.cb_languages.append_text(lang_in_cb)
            if lang.lower() == previous_lang:
                self.cb_languages.set_active(i)
            i += 1
    
    def _set_window_status(self, status):
        if status == 'error_initial_file':
            self.entry_sentence.set_sensitive(False)
            self.btn_lives.set_sensitive(False)
            self.btn_try_next.set_sensitive(False)
        elif status == 'new_sentence':
            self.btn_try_next.set_sensitive(True)
            self.entry_sentence.set_text('')
            self.img_state.show()
            self.btn_play.hide()
            self.img_state.set_from_stock('gtk-edit', 2)
            self.entry_sentence.set_sensitive(True)
            self.entry_sentence.set_tooltip_text('')
            self._set_statusbar_msg(_('Write the right sentence in the entry box'))
            if self.extra_lives > 0:
                self.btn_lives.set_sensitive(True)
            self.entry_sentence.grab_focus()
        elif status == 'complete_ok':
            self._set_statusbar_msg('')
            self.entry_sentence.set_sensitive(False)
            self.btn_lives.set_sensitive(False)
            self.btn_play.set_sensitive(True)
            self._set_statusbar_msg(_('Well done!! :) Try the next one'))
            self.img_state.hide()
            self.btn_play.show()
            self.btn_try_next.grab_focus()
        elif status == 'partial_ok':
            self._set_statusbar_msg('')
            self.img_state.set_from_stock('gtk-yes', 2)
        elif status == 'partial_no_ok':
            self._set_statusbar_msg('')
            self.img_state.set_from_stock('gtk-no', 2)
        elif status == 'empty_entry':
            self.img_state.set_from_stock('gtk-edit', 2)
        elif status == 'set_lbl_lives':
            if self.extra_lives == 0:
                self.btn_lives.set_sensitive(False)
                self.btn_lives.set_label(_('No Lives'))
            elif self.extra_lives == 1:
                self.btn_lives.set_label(_('1 Live'))
            elif self.extra_lives == 2:
                self.btn_lives.set_label(_('2 Lives'))
            elif self.extra_lives == 3:
                self.btn_lives.set_label(_('3 Lives'))
        elif status == 'used_live':
            self.btn_lives.set_sensitive(False)
            self.entry_sentence.set_sensitive(False)
            self.entry_sentence.set_text(self.sentences.get_current_solution())
            self.entry_sentence.set_tooltip_text(self.sentences.get_current_solution())
            self._set_statusbar_msg(_('That was the sentence!'))

    def _set_new_sentence(self):
        """Set initial random sentence"""
        self.sentences.set_sentence()
        self.lbl_subject.set_text(self.sentences.get_current_subject())
        self.lbl_verb.set_text(self.sentences.get_current_verb())
        self.lbl_tense.set_text(self.sentences.get_current_tense())
        self.lbl_option.set_text(self.sentences.get_current_option())
        self.lbl_object.set_text(self.sentences.get_current_object())
        self._set_window_status('new_sentence')
        
    def popup_error(self):
        dlg_error = Gtk.MessageDialog(None, 
                    Gtk.DialogFlags.MODAL | Gtk.DialogFlags.DESTROY_WITH_PARENT,
                    Gtk.MessageType.WARNING, Gtk.ButtonsType.CLOSE, 
                    "Error loading the sentences. Try to load another file")
        dlg_error.set_title("Error")
        dlg_error.run()
        dlg_error.destroy()
        return
        
    # GUI Events
    def on_entry_sentence_changed(self, widget):
        """Check sentence is completed"""
        if self.entry_sentence.get_text() == '':                               # empty
            self._set_window_status('empty_entry')
        elif self.sentences.check_is_ok(self.entry_sentence.get_text()):      # Check complete sentence!
            self._set_window_status('complete_ok')
        elif self.sentences.check_is_partial(self.entry_sentence.get_text()): # Check start sentence is right
            self._set_window_status('partial_ok')
        else:                                                                   # Wrong sentence
            self._set_window_status('partial_no_ok')
    
    def on_btn_try_next_clicked(self, widget):
        self._set_new_sentence()
    
    def on_menu_open_activate(self, widget, data=None):
        """Open sentences template"""
        dialog = Gtk.FileChooserDialog(_("Please choose a file"), None,
            Gtk.FileChooserAction.OPEN,
            (Gtk.STOCK_CANCEL, Gtk.ResponseType.CANCEL,
             Gtk.STOCK_SAVE, Gtk.ResponseType.OK))
        filter_xml = Gtk.FileFilter()
        filter_xml.set_name(_("XML files"))
        filter_xml.add_pattern("*.xml")
        dialog.add_filter(filter_xml)
        response = dialog.run()

        if response == Gtk.ResponseType.OK:
            dialog.hide()
            if self.sentences.load_file(dialog.get_filename()):
                self.extra_lives = 3
                self._set_window_status('set_lbl_lives')
                self._set_window_status('new_sentence')
                self._set_new_sentence()
            else:
                self.popup_error()
                
        dialog.destroy()

    def on_btn_lives_clicked(self, widget):
        """Extra lives"""
        self.extra_lives = self.extra_lives - 1
        self._set_window_status('set_lbl_lives')
        self._set_window_status('used_live')
    
    def on_btn_play_clicked(self, widget):
        """Play sentence"""
        sentence = ','.join(self.entry_sentence.get_text().split())
        sentence = sentence.replace("'", r"\'")
        self.btn_play.set_sensitive(False)
        
        if self.sentences.get_current_lang() == 'italian':
            speak = Process(target=speak_sentence, args=('it', sentence,))
        elif self.sentences.get_current_lang() == 'spanish':
            speak = Process(target=speak_sentence, args=('es', sentence,))
        else:
            speak = Process(target=speak_sentence, args=('en', sentence,))
        
        speak.start()
    
    def on_cb_languages_changed(self, widget, data=None):
        self.sentences.set_current_lang(self.translate_lang[self.cb_languages.get_active_text()])
        if (self.sentences.load_file()):
            self.utils.set_config_value('initial_language', self.translate_lang[self.cb_languages.get_active_text()])
            self._set_window_status('new_sentence')
            self._set_new_sentence()
        else:
            self._set_window_status('error_initial_file')
            self.popup_error()
            Gtk.main_quit()
    
    def on_main_delete_event(self, widget, data=None):
        Gtk.main_quit()
    def on_menu_close_activate(self, widget, data=None):
        Gtk.main_quit()
    
    def on_menu_about_activate(self, widget, data=None):
        about = WinAbout(self.winMain)
    
    def on_menu_web_activate(self, widget, data=None):
        webbrowser.open_new("http://pylang.tuxfamily.org")
    def on_menu_bugs_activate(self, widget, data=None):
        webbrowser.open_new("https://bugs.launchpad.net/pylang")
    def on_menu_donate_activate(self, widget, data=None):
        webbrowser.open_new("http://gufw.org/donate_pylang")
    
    def on_donate_hide_btn_clicked(self, widget, data=None):
        self.utils.set_config_value('hidedonatebtn', 'Yes')
        self.box_donate.hide()
    
    def _set_statusbar_msg(self, msg):
        cid = self.statusbar.get_context_id('default context')
        self.statusbar.push(cid, msg)
