# Copyright 1999 by Cayte Lindner.  All rights reserved.
# This code is part of the Biopython distribution and governed by its
# license.  Please see the LICENSE file that should have been included
# as part of this package.
#
# NOTE - This file has been split in two as a work arround for Jython JVM limits.

import os
import unittest

from Bio.ExPASy import Prosite


class TestPrositeRead(unittest.TestCase):

    def test_read1(self):
        "Parsing Prosite record ps00107.txt"
        filename = os.path.join('Prosite', 'ps00107.txt')
        handle = open(filename)
        record = Prosite.read(handle)
        handle.close()
        self.assertEqual(record.name, "PROTEIN_KINASE_ATP")
        self.assertEqual(record.type, "PATTERN")
        self.assertEqual(record.accession, "PS00107")
        self.assertEqual(record.created, "APR-1990")
        self.assertEqual(record.data_update, "NOV-1995")
        self.assertEqual(record.info_update, "MAR-2006")
        self.assertEqual(record.pdoc, "PDOC00100")
        self.assertEqual(record.description, "Protein kinases ATP-binding region signature.")
        self.assertEqual(record.pattern, "[LIV]-G-{P}-G-{P}-[FYWMGSTNH]-[SGA]-{PW}-[LIVCAT]-{PD}-x-[GSTACLIVMFY]-x(5,18)-[LIVMFYWCSTAR]-[AIVP]-[LIVMFAGCKR]-K.")
        self.assertEqual(record.matrix, [])
        self.assertEqual(record.rules, [])
        self.assertEqual(record.nr_sp_release, "49.3")
        self.assertEqual(record.nr_sp_seqs, 212425)
        self.assertEqual(record.cc_taxo_range, "??EPV")
        self.assertEqual(record.cc_max_repeat, "2")
        self.assertEqual(record.cc_site, [])
        self.read1_positive1(record)
        self.read1_positive2(record)
        self.read1_positive3(record)
        self.read1_false_neg(record)
        self.read1_false_pos(record)
        self.read1_potential(record)

    def read1_positive1(self, record):
        self.assertEqual(len(record.dr_positive), 1689)
        self.assertEqual(record.dr_positive[0], ('P13368', '7LESS_DROME'))
        self.assertEqual(record.dr_positive[1], ('P20806', '7LESS_DROVI'))
        self.assertEqual(record.dr_positive[2], ('P45894', 'AAPK1_CAEEL'))
        self.assertEqual(record.dr_positive[3], ('Q13131', 'AAPK1_HUMAN'))
        self.assertEqual(record.dr_positive[4], ('Q5EG47', 'AAPK1_MOUSE'))
        self.assertEqual(record.dr_positive[5], ('Q5RDH5', 'AAPK1_PONPY'))
        self.assertEqual(record.dr_positive[6], ('P54645', 'AAPK1_RAT'))
        self.assertEqual(record.dr_positive[7], ('Q95ZQ4', 'AAPK2_CAEEL'))
        self.assertEqual(record.dr_positive[8], ('P54646', 'AAPK2_HUMAN'))
        self.assertEqual(record.dr_positive[9], ('Q28948', 'AAPK2_PIG'))
        self.assertEqual(record.dr_positive[10], ('Q5RD00', 'AAPK2_PONPY'))
        self.assertEqual(record.dr_positive[11], ('Q09137', 'AAPK2_RAT'))
        self.assertEqual(record.dr_positive[12], ('P03949', 'ABL1_CAEEL'))
        self.assertEqual(record.dr_positive[13], ('P00519', 'ABL1_HUMAN'))
        self.assertEqual(record.dr_positive[14], ('P00520', 'ABL1_MOUSE'))
        self.assertEqual(record.dr_positive[15], ('P42684', 'ABL2_HUMAN'))
        self.assertEqual(record.dr_positive[16], ('P00522', 'ABL_DROME'))
        self.assertEqual(record.dr_positive[17], ('P10447', 'ABL_FSVHY'))
        self.assertEqual(record.dr_positive[18], ('P00521', 'ABL_MLVAB'))
        self.assertEqual(record.dr_positive[19], ('Q07912', 'ACK1_HUMAN'))
        self.assertEqual(record.dr_positive[20], ('O54967', 'ACK1_MOUSE'))
        self.assertEqual(record.dr_positive[21], ('O02742', 'ACK2_BOVIN'))
        self.assertEqual(record.dr_positive[22], ('P36896', 'ACV1B_HUMAN'))
        self.assertEqual(record.dr_positive[23], ('Q61271', 'ACV1B_MOUSE'))
        self.assertEqual(record.dr_positive[24], ('P80202', 'ACV1B_RAT'))
        self.assertEqual(record.dr_positive[25], ('Q8NER5', 'ACV1C_HUMAN'))
        self.assertEqual(record.dr_positive[26], ('Q8K348', 'ACV1C_MOUSE'))
        self.assertEqual(record.dr_positive[27], ('P70539', 'ACV1C_RAT'))
        self.assertEqual(record.dr_positive[28], ('P37023', 'ACVL1_HUMAN'))
        self.assertEqual(record.dr_positive[29], ('Q61288', 'ACVL1_MOUSE'))
        self.assertEqual(record.dr_positive[30], ('P80203', 'ACVL1_RAT'))
        self.assertEqual(record.dr_positive[31], ('Q28041', 'ACVR1_BOVIN'))
        self.assertEqual(record.dr_positive[32], ('Q04771', 'ACVR1_HUMAN'))
        self.assertEqual(record.dr_positive[33], ('P37172', 'ACVR1_MOUSE'))
        self.assertEqual(record.dr_positive[34], ('P80201', 'ACVR1_RAT'))
        self.assertEqual(record.dr_positive[35], ('P54741', 'AFSK_STRCO'))
        self.assertEqual(record.dr_positive[36], ('P54742', 'AFSK_STRGR'))
        self.assertEqual(record.dr_positive[37], ('Q01314', 'AKT1_BOVIN'))
        self.assertEqual(record.dr_positive[38], ('Q17941', 'AKT1_CAEEL'))
        self.assertEqual(record.dr_positive[39], ('Q8INB9', 'AKT1_DROME'))
        self.assertEqual(record.dr_positive[40], ('P31749', 'AKT1_HUMAN'))
        self.assertEqual(record.dr_positive[41], ('P31750', 'AKT1_MOUSE'))
        self.assertEqual(record.dr_positive[42], ('P47196', 'AKT1_RAT'))
        self.assertEqual(record.dr_positive[43], ('Q98TY9', 'AKT1_XENLA'))
        self.assertEqual(record.dr_positive[44], ('Q7ZX15', 'AKT2A_XENLA'))
        self.assertEqual(record.dr_positive[45], ('Q6IP76', 'AKT2B_XENLA'))
        self.assertEqual(record.dr_positive[46], ('Q9XTG7', 'AKT2_CAEEL'))
        self.assertEqual(record.dr_positive[47], ('P31751', 'AKT2_HUMAN'))
        self.assertEqual(record.dr_positive[48], ('Q60823', 'AKT2_MOUSE'))
        self.assertEqual(record.dr_positive[49], ('P47197', 'AKT2_RAT'))
        self.assertEqual(record.dr_positive[50], ('Q9Y243', 'AKT3_HUMAN'))
        self.assertEqual(record.dr_positive[51], ('Q9WUA6', 'AKT3_MOUSE'))
        self.assertEqual(record.dr_positive[52], ('Q63484', 'AKT3_RAT'))
        self.assertEqual(record.dr_positive[53], ('P31748', 'AKT_MLVAT'))
        self.assertEqual(record.dr_positive[54], ('Q96Q40', 'AL2S7_HUMAN'))
        self.assertEqual(record.dr_positive[55], ('Q9UM73', 'ALK_HUMAN'))
        self.assertEqual(record.dr_positive[56], ('P97793', 'ALK_MOUSE'))
        self.assertEqual(record.dr_positive[57], ('Q06548', 'APK1A_ARATH'))
        self.assertEqual(record.dr_positive[58], ('P46573', 'APK1B_ARATH'))
        self.assertEqual(record.dr_positive[59], ('P10398', 'ARAF_HUMAN'))
        self.assertEqual(record.dr_positive[60], ('P04627', 'ARAF_MOUSE'))
        self.assertEqual(record.dr_positive[61], ('O19004', 'ARAF_PIG'))
        self.assertEqual(record.dr_positive[62], ('P14056', 'ARAF_RAT'))
        self.assertEqual(record.dr_positive[63], ('P21146', 'ARBK1_BOVIN'))
        self.assertEqual(record.dr_positive[64], ('P25098', 'ARBK1_HUMAN'))
        self.assertEqual(record.dr_positive[65], ('Q64682', 'ARBK1_MESAU'))
        self.assertEqual(record.dr_positive[66], ('Q99MK8', 'ARBK1_MOUSE'))
        self.assertEqual(record.dr_positive[67], ('P26817', 'ARBK1_RAT'))
        self.assertEqual(record.dr_positive[68], ('P26818', 'ARBK2_BOVIN'))
        self.assertEqual(record.dr_positive[69], ('P35626', 'ARBK2_HUMAN'))
        self.assertEqual(record.dr_positive[70], ('P26819', 'ARBK2_RAT'))
        self.assertEqual(record.dr_positive[71], ('O59790', 'ARK1_SCHPO'))
        self.assertEqual(record.dr_positive[72], ('P43291', 'ASK1_ARATH'))
        self.assertEqual(record.dr_positive[73], ('P43292', 'ASK2_ARATH'))
        self.assertEqual(record.dr_positive[74], ('Q75CH3', 'ATG1_ASHGO'))
        self.assertEqual(record.dr_positive[75], ('Q4WPF2', 'ATG1_ASPFU'))
        self.assertEqual(record.dr_positive[76], ('Q6H9I1', 'ATG1_BOTCI'))
        self.assertEqual(record.dr_positive[77], ('Q5A649', 'ATG1_CANAL'))
        self.assertEqual(record.dr_positive[78], ('Q6FL58', 'ATG1_CANGA'))
        self.assertEqual(record.dr_positive[79], ('P87248', 'ATG1_COLLN'))
        self.assertEqual(record.dr_positive[80], ('Q5K8D3', 'ATG1_CRYNE'))
        self.assertEqual(record.dr_positive[81], ('Q6BS08', 'ATG1_DEBHA'))
        self.assertEqual(record.dr_positive[82], ('Q5BCU8', 'ATG1_EMENI'))
        self.assertEqual(record.dr_positive[83], ('Q6CSX2', 'ATG1_KLULA'))
        self.assertEqual(record.dr_positive[84], ('Q52EB3', 'ATG1_MAGGR'))
        self.assertEqual(record.dr_positive[85], ('Q7RX99', 'ATG1_NEUCR'))
        self.assertEqual(record.dr_positive[86], ('Q8TFN2', 'ATG1_PICAN'))
        self.assertEqual(record.dr_positive[87], ('Q8TGI1', 'ATG1_PICPA'))
        self.assertEqual(record.dr_positive[88], ('Q4P0K0', 'ATG1_USTMA'))
        self.assertEqual(record.dr_positive[89], ('Q6C7U0', 'ATG1_YARLI'))
        self.assertEqual(record.dr_positive[90], ('P53104', 'ATG1_YEAST'))
        self.assertEqual(record.dr_positive[91], ('Q96GD4', 'AURKB_HUMAN'))
        self.assertEqual(record.dr_positive[92], ('O70126', 'AURKB_MOUSE'))
        self.assertEqual(record.dr_positive[93], ('O55099', 'AURKB_RAT'))
        self.assertEqual(record.dr_positive[94], ('Q9UQB9', 'AURKC_HUMAN'))
        self.assertEqual(record.dr_positive[95], ('O88445', 'AURKC_MOUSE'))
        self.assertEqual(record.dr_positive[96], ('Q94F62', 'BAK1_ARATH'))
        self.assertEqual(record.dr_positive[97], ('Q01389', 'BCK1_YEAST'))
        self.assertEqual(record.dr_positive[98], ('P51451', 'BLK_HUMAN'))
        self.assertEqual(record.dr_positive[99], ('P16277', 'BLK_MOUSE'))
        self.assertEqual(record.dr_positive[100], ('Q13873', 'BMPR2_HUMAN'))
        self.assertEqual(record.dr_positive[101], ('O35607', 'BMPR2_MOUSE'))
        self.assertEqual(record.dr_positive[102], ('P36894', 'BMR1A_HUMAN'))
        self.assertEqual(record.dr_positive[103], ('P36895', 'BMR1A_MOUSE'))
        self.assertEqual(record.dr_positive[104], ('Q05438', 'BMR1B_CHICK'))
        self.assertEqual(record.dr_positive[105], ('O00238', 'BMR1B_HUMAN'))
        self.assertEqual(record.dr_positive[106], ('P36898', 'BMR1B_MOUSE'))
        self.assertEqual(record.dr_positive[107], ('P51813', 'BMX_HUMAN'))
        self.assertEqual(record.dr_positive[108], ('P97504', 'BMX_MOUSE'))
        self.assertEqual(record.dr_positive[109], ('Q04982', 'BRAF1_CHICK'))
        self.assertEqual(record.dr_positive[110], ('P34908', 'BRAF1_COTJA'))
        self.assertEqual(record.dr_positive[111], ('P15056', 'BRAF1_HUMAN'))
        self.assertEqual(record.dr_positive[112], ('P28028', 'BRAF1_MOUSE'))
        self.assertEqual(record.dr_positive[113], ('O22476', 'BRI1_ARATH'))
        self.assertEqual(record.dr_positive[114], ('Q8GUQ5', 'BRI1_LYCES'))
        self.assertEqual(record.dr_positive[115], ('Q8L899', 'BRI1_LYCPE'))
        self.assertEqual(record.dr_positive[116], ('Q9ZWC8', 'BRL1_ARATH'))
        self.assertEqual(record.dr_positive[117], ('Q9ZPS9', 'BRL2_ARATH'))
        self.assertEqual(record.dr_positive[118], ('Q9LJF3', 'BRL3_ARATH'))
        self.assertEqual(record.dr_positive[119], ('Q8TDC3', 'BRSK1_HUMAN'))
        self.assertEqual(record.dr_positive[120], ('Q8IWQ3', 'BRSK2_HUMAN'))
        self.assertEqual(record.dr_positive[121], ('P08630', 'BTKL_DROME'))
        self.assertEqual(record.dr_positive[122], ('Q8JH64', 'BTK_CHICK'))
        self.assertEqual(record.dr_positive[123], ('Q06187', 'BTK_HUMAN'))
        self.assertEqual(record.dr_positive[124], ('P35991', 'BTK_MOUSE'))
        self.assertEqual(record.dr_positive[125], ('O43683', 'BUB1_HUMAN'))
        self.assertEqual(record.dr_positive[126], ('O08901', 'BUB1_MOUSE'))
        self.assertEqual(record.dr_positive[127], ('P41695', 'BUB1_YEAST'))
        self.assertEqual(record.dr_positive[128], ('Q75D46', 'BUR1_ASHGO'))
        self.assertEqual(record.dr_positive[129], ('Q4WTN5', 'BUR1_ASPFU'))
        self.assertEqual(record.dr_positive[130], ('Q9Y7W4', 'BUR1_CANAL'))
        self.assertEqual(record.dr_positive[131], ('Q6FQ83', 'BUR1_CANGA'))
        self.assertEqual(record.dr_positive[132], ('Q6BV06', 'BUR1_DEBHA'))
        self.assertEqual(record.dr_positive[133], ('Q96VK3', 'BUR1_EMENI'))
        self.assertEqual(record.dr_positive[134], ('Q4I5U9', 'BUR1_GIBZE'))
        self.assertEqual(record.dr_positive[135], ('Q6CRA9', 'BUR1_KLULA'))
        self.assertEqual(record.dr_positive[136], ('Q871M9', 'BUR1_NEUCR'))
        self.assertEqual(record.dr_positive[137], ('Q6C842', 'BUR1_YARLI'))
        self.assertEqual(record.dr_positive[138], ('P23293', 'BUR1_YEAST'))
        self.assertEqual(record.dr_positive[139], ('P10506', 'BYR1_SCHPO'))
        self.assertEqual(record.dr_positive[140], ('P28829', 'BYR2_SCHPO'))
        self.assertEqual(record.dr_positive[141], ('Q9VBW3', 'CAD96_DROME'))
        self.assertEqual(record.dr_positive[142], ('Q754N7', 'CBK1_ASHGO'))
        self.assertEqual(record.dr_positive[143], ('Q6FP74', 'CBK1_CANGA'))
        self.assertEqual(record.dr_positive[144], ('Q6BLJ9', 'CBK1_DEBHA'))
        self.assertEqual(record.dr_positive[145], ('P31034', 'CBK1_KLULA'))
        self.assertEqual(record.dr_positive[146], ('Q6TGC6', 'CBK1_PNECA'))
        self.assertEqual(record.dr_positive[147], ('Q6CFS5', 'CBK1_YARLI'))
        self.assertEqual(record.dr_positive[148], ('P53894', 'CBK1_YEAST'))
        self.assertEqual(record.dr_positive[149], ('P38973', 'CC2H1_TRYBB'))
        self.assertEqual(record.dr_positive[150], ('P54664', 'CC2H1_TRYCO'))
        self.assertEqual(record.dr_positive[151], ('P54665', 'CC2H2_TRYBB'))
        self.assertEqual(record.dr_positive[152], ('P54666', 'CC2H3_TRYBB'))
        self.assertEqual(record.dr_positive[153], ('Q43531', 'CCAMK_LILLO'))
        self.assertEqual(record.dr_positive[154], ('Q6RET7', 'CCAMK_MEDTR'))
        self.assertEqual(record.dr_positive[155], ('Q8IZL9', 'CCRK_HUMAN'))
        self.assertEqual(record.dr_positive[156], ('Q5EDC3', 'CCRK_MACMU'))
        self.assertEqual(record.dr_positive[157], ('Q9JHU3', 'CCRK_MOUSE'))
        self.assertEqual(record.dr_positive[158], ('Q5R7I7', 'CCRK_PONPY'))
        self.assertEqual(record.dr_positive[159], ('Q4KM34', 'CCRK_RAT'))
        self.assertEqual(record.dr_positive[160], ('Q14004', 'CD2L5_HUMAN'))
        self.assertEqual(record.dr_positive[161], ('Q69ZA1', 'CD2L5_MOUSE'))
        self.assertEqual(record.dr_positive[162], ('Q9BWU1', 'CD2L6_HUMAN'))
        self.assertEqual(record.dr_positive[163], ('Q8BWD8', 'CD2L6_MOUSE'))
        self.assertEqual(record.dr_positive[164], ('P46551', 'CD2L7_CAEEL'))
        self.assertEqual(record.dr_positive[165], ('Q9NYV4', 'CD2L7_HUMAN'))
        self.assertEqual(record.dr_positive[166], ('P27636', 'CDC15_YEAST'))
        self.assertEqual(record.dr_positive[167], ('P24923', 'CDC21_MEDSA'))
        self.assertEqual(record.dr_positive[168], ('P29618', 'CDC21_ORYSA'))
        self.assertEqual(record.dr_positive[169], ('P35567', 'CDC21_XENLA'))
        self.assertEqual(record.dr_positive[170], ('Q05006', 'CDC22_MEDSA'))
        self.assertEqual(record.dr_positive[171], ('P29619', 'CDC22_ORYSA'))
        self.assertEqual(record.dr_positive[172], ('P24033', 'CDC22_XENLA'))
        self.assertEqual(record.dr_positive[173], ('P43063', 'CDC28_CANAL'))
        self.assertEqual(record.dr_positive[174], ('P00546', 'CDC28_YEAST'))
        self.assertEqual(record.dr_positive[175], ('Q38772', 'CDC2A_ANTMA'))
        self.assertEqual(record.dr_positive[176], ('P24100', 'CDC2A_ARATH'))
        self.assertEqual(record.dr_positive[177], ('P25859', 'CDC2B_ARATH'))
        self.assertEqual(record.dr_positive[178], ('Q38774', 'CDC2C_ANTMA'))
        self.assertEqual(record.dr_positive[179], ('P23573', 'CDC2C_DROME'))
        self.assertEqual(record.dr_positive[180], ('Q38775', 'CDC2D_ANTMA'))
        self.assertEqual(record.dr_positive[181], ('Q01917', 'CDC2H_CRIFA'))
        self.assertEqual(record.dr_positive[182], ('P34117', 'CDC2H_DICDI'))
        self.assertEqual(record.dr_positive[183], ('P61075', 'CDC2H_PLAF7'))
        self.assertEqual(record.dr_positive[184], ('Q07785', 'CDC2H_PLAFK'))
        self.assertEqual(record.dr_positive[185], ('P54119', 'CDC2_AJECA'))
        self.assertEqual(record.dr_positive[186], ('P48734', 'CDC2_BOVIN'))
        self.assertEqual(record.dr_positive[187], ('P34556', 'CDC2_CAEEL'))
        self.assertEqual(record.dr_positive[188], ('P51958', 'CDC2_CARAU'))
        self.assertEqual(record.dr_positive[189], ('P93101', 'CDC2_CHERU'))
        self.assertEqual(record.dr_positive[190], ('P13863', 'CDC2_CHICK'))
        self.assertEqual(record.dr_positive[191], ('P34112', 'CDC2_DICDI'))
        self.assertEqual(record.dr_positive[192], ('P23572', 'CDC2_DROME'))
        self.assertEqual(record.dr_positive[193], ('Q00646', 'CDC2_EMENI'))
        self.assertEqual(record.dr_positive[194], ('P06493', 'CDC2_HUMAN'))
        self.assertEqual(record.dr_positive[195], ('P23111', 'CDC2_MAIZE'))
        self.assertEqual(record.dr_positive[196], ('P11440', 'CDC2_MOUSE'))
        self.assertEqual(record.dr_positive[197], ('Q9DGA5', 'CDC2_ORYCU'))
        self.assertEqual(record.dr_positive[198], ('Q9DGA2', 'CDC2_ORYJA'))
        self.assertEqual(record.dr_positive[199], ('Q9DGD3', 'CDC2_ORYLA'))
        self.assertEqual(record.dr_positive[200], ('Q9DG98', 'CDC2_ORYLU'))
        self.assertEqual(record.dr_positive[201], ('Q9W739', 'CDC2_RANDY'))
        self.assertEqual(record.dr_positive[202], ('P39951', 'CDC2_RAT'))
        self.assertEqual(record.dr_positive[203], ('P04551', 'CDC2_SCHPO'))
        self.assertEqual(record.dr_positive[204], ('Q41639', 'CDC2_VIGAC'))
        self.assertEqual(record.dr_positive[205], ('P52389', 'CDC2_VIGUN'))
        self.assertEqual(record.dr_positive[206], ('P32562', 'CDC5_YEAST'))
        self.assertEqual(record.dr_positive[207], ('O00311', 'CDC7_HUMAN'))
        self.assertEqual(record.dr_positive[208], ('Q9Z0H0', 'CDC7_MOUSE'))
        self.assertEqual(record.dr_positive[209], ('P41892', 'CDC7_SCHPO'))
        self.assertEqual(record.dr_positive[210], ('Q15131', 'CDK10_HUMAN'))
        self.assertEqual(record.dr_positive[211], ('Q5E9Y0', 'CDK2_BOVIN'))
        self.assertEqual(record.dr_positive[212], ('P43450', 'CDK2_CARAU'))
        self.assertEqual(record.dr_positive[213], ('O55076', 'CDK2_CRIGR'))
        self.assertEqual(record.dr_positive[214], ('Q04770', 'CDK2_ENTHI'))
        self.assertEqual(record.dr_positive[215], ('P24941', 'CDK2_HUMAN'))
        self.assertEqual(record.dr_positive[216], ('P48963', 'CDK2_MESAU'))
        self.assertEqual(record.dr_positive[217], ('P97377', 'CDK2_MOUSE'))
        self.assertEqual(record.dr_positive[218], ('Q63699', 'CDK2_RAT'))
        self.assertEqual(record.dr_positive[219], ('P23437', 'CDK2_XENLA'))
        self.assertEqual(record.dr_positive[220], ('Q00526', 'CDK3_HUMAN'))
        self.assertEqual(record.dr_positive[221], ('Q80YP0', 'CDK3_MOUSE'))
        self.assertEqual(record.dr_positive[222], ('P11802', 'CDK4_HUMAN'))
        self.assertEqual(record.dr_positive[223], ('P30285', 'CDK4_MOUSE'))
        self.assertEqual(record.dr_positive[224], ('P79432', 'CDK4_PIG'))
        self.assertEqual(record.dr_positive[225], ('P35426', 'CDK4_RAT'))
        self.assertEqual(record.dr_positive[226], ('Q91727', 'CDK4_XENLA'))
        self.assertEqual(record.dr_positive[227], ('Q02399', 'CDK5_BOVIN'))
        self.assertEqual(record.dr_positive[228], ('P48609', 'CDK5_DROME'))
        self.assertEqual(record.dr_positive[229], ('Q00535', 'CDK5_HUMAN'))
        self.assertEqual(record.dr_positive[230], ('P49615', 'CDK5_MOUSE'))
        self.assertEqual(record.dr_positive[231], ('Q03114', 'CDK5_RAT'))
        self.assertEqual(record.dr_positive[232], ('P51166', 'CDK5_XENLA'))
        self.assertEqual(record.dr_positive[233], ('Q00534', 'CDK6_HUMAN'))
        self.assertEqual(record.dr_positive[234], ('Q64261', 'CDK6_MOUSE'))
        self.assertEqual(record.dr_positive[235], ('P51953', 'CDK7_CARAU'))
        self.assertEqual(record.dr_positive[236], ('P54685', 'CDK7_DICDI'))
        self.assertEqual(record.dr_positive[237], ('P50613', 'CDK7_HUMAN'))
        self.assertEqual(record.dr_positive[238], ('Q03147', 'CDK7_MOUSE'))
        self.assertEqual(record.dr_positive[239], ('P51952', 'CDK7_RAT'))
        self.assertEqual(record.dr_positive[240], ('P20911', 'CDK7_XENLA'))
        self.assertEqual(record.dr_positive[241], ('Q9VT57', 'CDK8_DROME'))
        self.assertEqual(record.dr_positive[242], ('P49336', 'CDK8_HUMAN'))
        self.assertEqual(record.dr_positive[243], ('Q4V862', 'CDK9A_XENLA'))
        self.assertEqual(record.dr_positive[244], ('Q7ZX42', 'CDK9B_XENLA'))
        self.assertEqual(record.dr_positive[245], ('Q5EAB2', 'CDK9_BOVIN'))
        self.assertEqual(record.dr_positive[246], ('Q9TVL3', 'CDK9_CAEEL'))
        self.assertEqual(record.dr_positive[247], ('Q5ZKN1', 'CDK9_CHICK'))
        self.assertEqual(record.dr_positive[248], ('P50750', 'CDK9_HUMAN'))
        self.assertEqual(record.dr_positive[249], ('Q99J95', 'CDK9_MOUSE'))
        self.assertEqual(record.dr_positive[250], ('Q641Z4', 'CDK9_RAT'))
        self.assertEqual(record.dr_positive[251], ('Q96WV9', 'CDK9_SCHPO'))
        self.assertEqual(record.dr_positive[252], ('Q6GLD8', 'CDK9_XENTR'))
        self.assertEqual(record.dr_positive[253], ('Q6AXJ9', 'CDKL1_BRARE'))
        self.assertEqual(record.dr_positive[254], ('Q00532', 'CDKL1_HUMAN'))
        self.assertEqual(record.dr_positive[255], ('Q8CEQ0', 'CDKL1_MOUSE'))
        self.assertEqual(record.dr_positive[256], ('Q66HE7', 'CDKL1_RAT'))
        self.assertEqual(record.dr_positive[257], ('Q92772', 'CDKL2_HUMAN'))
        self.assertEqual(record.dr_positive[258], ('Q4R7T5', 'CDKL2_MACFA'))
        self.assertEqual(record.dr_positive[259], ('Q9QUK0', 'CDKL2_MOUSE'))
        self.assertEqual(record.dr_positive[260], ('Q5R754', 'CDKL2_PONPY'))
        self.assertEqual(record.dr_positive[261], ('Q9TTK0', 'CDKL2_RABIT'))
        self.assertEqual(record.dr_positive[262], ('Q5XIT0', 'CDKL2_RAT'))
        self.assertEqual(record.dr_positive[263], ('Q8IVW4', 'CDKL3_HUMAN'))
        self.assertEqual(record.dr_positive[264], ('Q4R8T9', 'CDKL3_MACFA'))
        self.assertEqual(record.dr_positive[265], ('Q8BLF2', 'CDKL3_MOUSE'))
        self.assertEqual(record.dr_positive[266], ('Q9JM01', 'CDKL3_RAT'))
        self.assertEqual(record.dr_positive[267], ('Q3TZA2', 'CDKL4_MOUSE'))
        self.assertEqual(record.dr_positive[268], ('O76039', 'CDKL5_HUMAN'))
        self.assertEqual(record.dr_positive[269], ('Q06850', 'CDPK1_ARATH'))
        self.assertEqual(record.dr_positive[270], ('P53682', 'CDPK1_ORYSA'))
        self.assertEqual(record.dr_positive[271], ('P62344', 'CDPK1_PLAF7'))
        self.assertEqual(record.dr_positive[272], ('P62343', 'CDPK1_PLAFK'))
        self.assertEqual(record.dr_positive[273], ('Q7RAH3', 'CDPK1_PLAYO'))
        self.assertEqual(record.dr_positive[274], ('P49101', 'CDPK2_MAIZE'))
        self.assertEqual(record.dr_positive[275], ('P53683', 'CDPK2_ORYSA'))
        self.assertEqual(record.dr_positive[276], ('Q8ICR0', 'CDPK2_PLAF7'))
        self.assertEqual(record.dr_positive[277], ('O15865', 'CDPK2_PLAFK'))
        self.assertEqual(record.dr_positive[278], ('P53684', 'CDPK3_ORYSA'))
        self.assertEqual(record.dr_positive[279], ('Q9NJU9', 'CDPK3_PLAF7'))
        self.assertEqual(record.dr_positive[280], ('Q7RAV5', 'CDPK3_PLAYO'))
        self.assertEqual(record.dr_positive[281], ('P62345', 'CDPK4_PLABA'))
        self.assertEqual(record.dr_positive[282], ('Q8IBS5', 'CDPK4_PLAF7'))
        self.assertEqual(record.dr_positive[283], ('Q7RJG2', 'CDPK4_PLAYO'))
        self.assertEqual(record.dr_positive[284], ('P28582', 'CDPK_DAUCA'))
        self.assertEqual(record.dr_positive[285], ('P28583', 'CDPK_SOYBN'))
        self.assertEqual(record.dr_positive[286], ('P07334', 'CDR1_SCHPO'))
        self.assertEqual(record.dr_positive[287], ('Q09170', 'CDS1_SCHPO'))
        self.assertEqual(record.dr_positive[288], ('P18460', 'CEK2_CHICK'))
        self.assertEqual(record.dr_positive[289], ('P18461', 'CEK3_CHICK'))
        self.assertEqual(record.dr_positive[290], ('Q61RA2', 'CHK1_CAEBR'))
        self.assertEqual(record.dr_positive[291], ('Q9N3Z3', 'CHK1_CAEEL'))
        self.assertEqual(record.dr_positive[292], ('Q8AYC9', 'CHK1_CHICK'))
        self.assertEqual(record.dr_positive[293], ('O61661', 'CHK1_DROME'))
        self.assertEqual(record.dr_positive[294], ('O14757', 'CHK1_HUMAN'))
        self.assertEqual(record.dr_positive[295], ('O35280', 'CHK1_MOUSE'))
        self.assertEqual(record.dr_positive[296], ('Q91ZN7', 'CHK1_RAT'))
        self.assertEqual(record.dr_positive[297], ('P34208', 'CHK1_SCHPO'))
        self.assertEqual(record.dr_positive[298], ('Q6DE87', 'CHK1_XENLA'))
        self.assertEqual(record.dr_positive[299], ('P38147', 'CHK1_YEAST'))
        self.assertEqual(record.dr_positive[300], ('Q9U1Y5', 'CHK2_CAEEL'))
        self.assertEqual(record.dr_positive[301], ('Q8RWC9', 'CIPK1_ARATH'))
        self.assertEqual(record.dr_positive[302], ('Q6X4A2', 'CIPK1_ORYSA'))
        self.assertEqual(record.dr_positive[303], ('P40233', 'CKI1_SCHPO'))
        self.assertEqual(record.dr_positive[304], ('P40234', 'CKI2_SCHPO'))
        self.assertEqual(record.dr_positive[305], ('O74135', 'CKI3_SCHPO'))
        self.assertEqual(record.dr_positive[306], ('P49759', 'CLK1_HUMAN'))
        self.assertEqual(record.dr_positive[307], ('P22518', 'CLK1_MOUSE'))
        self.assertEqual(record.dr_positive[308], ('P49760', 'CLK2_HUMAN'))
        self.assertEqual(record.dr_positive[309], ('O35491', 'CLK2_MOUSE'))
        self.assertEqual(record.dr_positive[310], ('P49761', 'CLK3_HUMAN'))
        self.assertEqual(record.dr_positive[311], ('O35492', 'CLK3_MOUSE'))
        self.assertEqual(record.dr_positive[312], ('Q63117', 'CLK3_RAT'))
        self.assertEqual(record.dr_positive[313], ('Q9HAZ1', 'CLK4_HUMAN'))
        self.assertEqual(record.dr_positive[314], ('O35493', 'CLK4_MOUSE'))
        self.assertEqual(record.dr_positive[315], ('P38679', 'COT1_NEUCR'))
        self.assertEqual(record.dr_positive[316], ('O22932', 'CPK11_ARATH'))
        self.assertEqual(record.dr_positive[317], ('P92937', 'CPK15_ARATH'))
        self.assertEqual(record.dr_positive[318], ('Q8NK05', 'CPK1_CRYNE'))
        self.assertEqual(record.dr_positive[319], ('Q9LDI3', 'CPK24_ARATH'))
        self.assertEqual(record.dr_positive[320], ('O24585', 'CRI4_MAIZE'))
        self.assertEqual(record.dr_positive[321], ('Q06309', 'CRK1_LEIME'))
        self.assertEqual(record.dr_positive[322], ('Q12126', 'CRK1_SCHPO'))
        self.assertEqual(record.dr_positive[323], ('P53681', 'CRK_DAUCA'))
        self.assertEqual(record.dr_positive[324], ('P13369', 'CSF1R_FELCA'))
        self.assertEqual(record.dr_positive[325], ('P07333', 'CSF1R_HUMAN'))
        self.assertEqual(record.dr_positive[326], ('P09581', 'CSF1R_MOUSE'))
        self.assertEqual(record.dr_positive[327], ('Q00495', 'CSF1R_RAT'))
        self.assertEqual(record.dr_positive[328], ('Q08467', 'CSK21_ARATH'))
        self.assertEqual(record.dr_positive[329], ('P68399', 'CSK21_BOVIN'))
        self.assertEqual(record.dr_positive[330], ('P21868', 'CSK21_CHICK'))
        self.assertEqual(record.dr_positive[331], ('P68400', 'CSK21_HUMAN'))
        self.assertEqual(record.dr_positive[332], ('Q60737', 'CSK21_MOUSE'))
        self.assertEqual(record.dr_positive[333], ('P33674', 'CSK21_RABIT'))
        self.assertEqual(record.dr_positive[334], ('P19139', 'CSK21_RAT'))
        self.assertEqual(record.dr_positive[335], ('P15790', 'CSK21_YEAST'))
        self.assertEqual(record.dr_positive[336], ('Q08466', 'CSK22_ARATH'))
        self.assertEqual(record.dr_positive[337], ('P20427', 'CSK22_BOVIN'))
        self.assertEqual(record.dr_positive[338], ('P21869', 'CSK22_CHICK'))
        self.assertEqual(record.dr_positive[339], ('P19784', 'CSK22_HUMAN'))
        self.assertEqual(record.dr_positive[340], ('O54833', 'CSK22_MOUSE'))
        self.assertEqual(record.dr_positive[341], ('P28020', 'CSK22_XENLA'))
        self.assertEqual(record.dr_positive[342], ('P19454', 'CSK22_YEAST'))
        self.assertEqual(record.dr_positive[343], ('O64817', 'CSK23_ARATH'))
        self.assertEqual(record.dr_positive[344], ('P18334', 'CSK2A_CAEEL'))
        self.assertEqual(record.dr_positive[345], ('Q02720', 'CSK2A_DICDI'))
        self.assertEqual(record.dr_positive[346], ('P08181', 'CSK2A_DROME'))
        self.assertEqual(record.dr_positive[347], ('P28523', 'CSK2A_MAIZE'))
        self.assertEqual(record.dr_positive[348], ('Q8TG13', 'CSK2A_NEUCR'))
        self.assertEqual(record.dr_positive[349], ('P40231', 'CSK2A_SCHPO'))
        self.assertEqual(record.dr_positive[350], ('O76484', 'CSK2A_SPOFR'))
        self.assertEqual(record.dr_positive[351], ('P28547', 'CSK2A_THEPA'))
        self.assertEqual(record.dr_positive[352], ('P41239', 'CSK_CHICK'))
        self.assertEqual(record.dr_positive[353], ('P41240', 'CSK_HUMAN'))
        self.assertEqual(record.dr_positive[354], ('P41241', 'CSK_MOUSE'))
        self.assertEqual(record.dr_positive[355], ('P32577', 'CSK_RAT'))
        self.assertEqual(record.dr_positive[356], ('Q03957', 'CTK1_YEAST'))
        self.assertEqual(record.dr_positive[357], ('Q05609', 'CTR1_ARATH'))
        self.assertEqual(record.dr_positive[358], ('O14578', 'CTRO_HUMAN'))
        self.assertEqual(record.dr_positive[359], ('P49025', 'CTRO_MOUSE'))
        self.assertEqual(record.dr_positive[360], ('P27450', 'CX32_ARATH'))
        self.assertEqual(record.dr_positive[361], ('P20792', 'DAF1_CAEEL'))
        self.assertEqual(record.dr_positive[362], ('P53355', 'DAPK1_HUMAN'))
        self.assertEqual(record.dr_positive[363], ('Q80YE7', 'DAPK1_MOUSE'))
        self.assertEqual(record.dr_positive[364], ('Q9UIK4', 'DAPK2_HUMAN'))
        self.assertEqual(record.dr_positive[365], ('Q8VDF3', 'DAPK2_MOUSE'))
        self.assertEqual(record.dr_positive[366], ('O43293', 'DAPK3_HUMAN'))
        self.assertEqual(record.dr_positive[367], ('O54784', 'DAPK3_MOUSE'))
        self.assertEqual(record.dr_positive[368], ('O88764', 'DAPK3_RAT'))
        self.assertEqual(record.dr_positive[369], ('P32328', 'DBF20_YEAST'))
        self.assertEqual(record.dr_positive[370], ('P22204', 'DBF2_YEAST'))
        self.assertEqual(record.dr_positive[371], ('O15075', 'DCAK1_HUMAN'))
        self.assertEqual(record.dr_positive[372], ('Q9JLM8', 'DCAK1_MOUSE'))
        self.assertEqual(record.dr_positive[373], ('O08875', 'DCAK1_RAT'))
        self.assertEqual(record.dr_positive[374], ('Q8N568', 'DCAK2_HUMAN'))
        self.assertEqual(record.dr_positive[375], ('Q6PGN3', 'DCAK2_MOUSE'))
        self.assertEqual(record.dr_positive[376], ('O76324', 'DCO_DROME'))
        self.assertEqual(record.dr_positive[377], ('Q09013', 'DMPK_HUMAN'))
        self.assertEqual(record.dr_positive[378], ('P54265', 'DMPK_MOUSE'))
        self.assertEqual(record.dr_positive[379], ('P49762', 'DOA_DROME'))
        self.assertEqual(record.dr_positive[380], ('P36616', 'DSK1_SCHPO'))
        self.assertEqual(record.dr_positive[381], ('Q24324', 'DSOR1_DROME'))
        self.assertEqual(record.dr_positive[382], ('Q9Y2A5', 'DUET_HUMAN'))
        self.assertEqual(record.dr_positive[383], ('P39009', 'DUN1_YEAST'))
        self.assertEqual(record.dr_positive[384], ('Q13627', 'DYR1A_HUMAN'))
        self.assertEqual(record.dr_positive[385], ('Q61214', 'DYR1A_MOUSE'))
        self.assertEqual(record.dr_positive[386], ('Q63470', 'DYR1A_RAT'))
        self.assertEqual(record.dr_positive[387], ('Q9Y463', 'DYR1B_HUMAN'))
        self.assertEqual(record.dr_positive[388], ('Q9Z188', 'DYR1B_MOUSE'))
        self.assertEqual(record.dr_positive[389], ('Q9V3D5', 'DYRK2_DROME'))
        self.assertEqual(record.dr_positive[390], ('Q92630', 'DYRK2_HUMAN'))
        self.assertEqual(record.dr_positive[391], ('P83102', 'DYRK3_DROME'))
        self.assertEqual(record.dr_positive[392], ('O43781', 'DYRK3_HUMAN'))
        self.assertEqual(record.dr_positive[393], ('Q9NR20', 'DYRK4_HUMAN'))
        self.assertEqual(record.dr_positive[394], ('Q9BQI3', 'E2AK1_HUMAN'))
        self.assertEqual(record.dr_positive[395], ('Q9Z2R9', 'E2AK1_MOUSE'))
        self.assertEqual(record.dr_positive[396], ('P33279', 'E2AK1_RABIT'))
        self.assertEqual(record.dr_positive[397], ('Q63185', 'E2AK1_RAT'))
        self.assertEqual(record.dr_positive[398], ('O13889', 'E2AK1_SCHPO'))
        self.assertEqual(record.dr_positive[399], ('P19525', 'E2AK2_HUMAN'))
        self.assertEqual(record.dr_positive[400], ('Q03963', 'E2AK2_MOUSE'))
        self.assertEqual(record.dr_positive[401], ('Q9UTE5', 'E2AK2_SCHPO'))
        self.assertEqual(record.dr_positive[402], ('Q19192', 'E2AK3_CAEEL'))
        self.assertEqual(record.dr_positive[403], ('Q9NIV1', 'E2AK3_DROME'))
        self.assertEqual(record.dr_positive[404], ('Q9NZJ5', 'E2AK3_HUMAN'))
        self.assertEqual(record.dr_positive[405], ('Q9P2K8', 'E2AK4_HUMAN'))
        self.assertEqual(record.dr_positive[406], ('Q9QZ05', 'E2AK4_MOUSE'))
        self.assertEqual(record.dr_positive[407], ('P04412', 'EGFR_DROME'))
        self.assertEqual(record.dr_positive[408], ('P00533', 'EGFR_HUMAN'))
        self.assertEqual(record.dr_positive[409], ('Q01279', 'EGFR_MOUSE'))
        self.assertEqual(record.dr_positive[410], ('Q10656', 'EGL15_CAEEL'))
        self.assertEqual(record.dr_positive[411], ('O13148', 'EPA4A_BRARE'))
        self.assertEqual(record.dr_positive[412], ('Q91845', 'EPA4A_XENLA'))
        self.assertEqual(record.dr_positive[413], ('Q91694', 'EPA4B_XENLA'))
        self.assertEqual(record.dr_positive[414], ('Q91571', 'EPB1A_XENLA'))
        self.assertEqual(record.dr_positive[415], ('Q91736', 'EPB1B_XENLA'))
        self.assertEqual(record.dr_positive[416], ('P21709', 'EPHA1_HUMAN'))
        self.assertEqual(record.dr_positive[417], ('Q60750', 'EPHA1_MOUSE'))
        self.assertEqual(record.dr_positive[418], ('P29317', 'EPHA2_HUMAN'))
        self.assertEqual(record.dr_positive[419], ('Q03145', 'EPHA2_MOUSE'))
        self.assertEqual(record.dr_positive[420], ('O13146', 'EPHA3_BRARE'))
        self.assertEqual(record.dr_positive[421], ('P29318', 'EPHA3_CHICK'))
        self.assertEqual(record.dr_positive[422], ('P29320', 'EPHA3_HUMAN'))
        self.assertEqual(record.dr_positive[423], ('P29319', 'EPHA3_MOUSE'))
        self.assertEqual(record.dr_positive[424], ('O08680', 'EPHA3_RAT'))
        self.assertEqual(record.dr_positive[425], ('Q07496', 'EPHA4_CHICK'))
        self.assertEqual(record.dr_positive[426], ('P54764', 'EPHA4_HUMAN'))
        self.assertEqual(record.dr_positive[427], ('Q03137', 'EPHA4_MOUSE'))
        self.assertEqual(record.dr_positive[428], ('P54755', 'EPHA5_CHICK'))
        self.assertEqual(record.dr_positive[429], ('P54756', 'EPHA5_HUMAN'))
        self.assertEqual(record.dr_positive[430], ('Q60629', 'EPHA5_MOUSE'))
        self.assertEqual(record.dr_positive[431], ('Q62413', 'EPHA6_MOUSE'))
        self.assertEqual(record.dr_positive[432], ('P54758', 'EPHA6_RAT'))
        self.assertEqual(record.dr_positive[433], ('O42422', 'EPHA7_CHICK'))
        self.assertEqual(record.dr_positive[434], ('Q15375', 'EPHA7_HUMAN'))
        self.assertEqual(record.dr_positive[435], ('Q61772', 'EPHA7_MOUSE'))
        self.assertEqual(record.dr_positive[436], ('P54759', 'EPHA7_RAT'))
        self.assertEqual(record.dr_positive[437], ('P29322', 'EPHA8_HUMAN'))
        self.assertEqual(record.dr_positive[438], ('O09127', 'EPHA8_MOUSE'))
        self.assertEqual(record.dr_positive[439], ('P29321', 'EPHA8_RAT'))
        self.assertEqual(record.dr_positive[440], ('Q07494', 'EPHB1_CHICK'))
        self.assertEqual(record.dr_positive[441], ('P54762', 'EPHB1_HUMAN'))
        self.assertEqual(record.dr_positive[442], ('P09759', 'EPHB1_RAT'))
        self.assertEqual(record.dr_positive[443], ('P28693', 'EPHB2_CHICK'))
        self.assertEqual(record.dr_positive[444], ('Q90344', 'EPHB2_COTJA'))
        self.assertEqual(record.dr_positive[445], ('P29323', 'EPHB2_HUMAN'))
        self.assertEqual(record.dr_positive[446], ('P54763', 'EPHB2_MOUSE'))
        self.assertEqual(record.dr_positive[447], ('O13147', 'EPHB3_BRARE'))
        self.assertEqual(record.dr_positive[448], ('Q07498', 'EPHB3_CHICK'))
        self.assertEqual(record.dr_positive[449], ('P54753', 'EPHB3_HUMAN'))
        self.assertEqual(record.dr_positive[450], ('P54754', 'EPHB3_MOUSE'))
        self.assertEqual(record.dr_positive[451], ('Q91735', 'EPHB3_XENLA'))
        self.assertEqual(record.dr_positive[452], ('P54760', 'EPHB4_HUMAN'))
        self.assertEqual(record.dr_positive[453], ('P54761', 'EPHB4_MOUSE'))
        self.assertEqual(record.dr_positive[454], ('Q07497', 'EPHB5_CHICK'))
        self.assertEqual(record.dr_positive[455], ('O18735', 'ERBB2_CANFA'))
        self.assertEqual(record.dr_positive[456], ('P04626', 'ERBB2_HUMAN'))
        self.assertEqual(record.dr_positive[457], ('Q60553', 'ERBB2_MESAU'))
        self.assertEqual(record.dr_positive[458], ('P70424', 'ERBB2_MOUSE'))
        self.assertEqual(record.dr_positive[459], ('P06494', 'ERBB2_RAT'))
        self.assertEqual(record.dr_positive[460], ('Q15303', 'ERBB4_HUMAN'))
        self.assertEqual(record.dr_positive[461], ('Q62956', 'ERBB4_RAT'))
        self.assertEqual(record.dr_positive[462], ('P00534', 'ERBB_ALV'))
        self.assertEqual(record.dr_positive[463], ('P00535', 'ERBB_AVIER'))
        self.assertEqual(record.dr_positive[464], ('P11273', 'ERBB_AVIEU'))
        self.assertEqual(record.dr_positive[465], ('P28869', 'ERK1_CANAL'))
        self.assertEqual(record.dr_positive[466], ('P42525', 'ERK1_DICDI'))
        self.assertEqual(record.dr_positive[467], ('P40417', 'ERKA_DROME'))
        self.assertEqual(record.dr_positive[468], ('Q9LYN8', 'EXS_ARATH'))
        self.assertEqual(record.dr_positive[469], ('Q00944', 'FAK1_CHICK'))
        self.assertEqual(record.dr_positive[470], ('Q05397', 'FAK1_HUMAN'))
        self.assertEqual(record.dr_positive[471], ('P34152', 'FAK1_MOUSE'))
        self.assertEqual(record.dr_positive[472], ('O35346', 'FAK1_RAT'))
        self.assertEqual(record.dr_positive[473], ('Q91738', 'FAK1_XENLA'))
        self.assertEqual(record.dr_positive[474], ('Q14289', 'FAK2_HUMAN'))
        self.assertEqual(record.dr_positive[475], ('Q9QVP9', 'FAK2_MOUSE'))
        self.assertEqual(record.dr_positive[476], ('P70600', 'FAK2_RAT'))
        self.assertEqual(record.dr_positive[477], ('P16591', 'FER_HUMAN'))
        self.assertEqual(record.dr_positive[478], ('P14238', 'FES_FELCA'))
        self.assertEqual(record.dr_positive[479], ('P00542', 'FES_FSVGA'))
        self.assertEqual(record.dr_positive[480], ('P00543', 'FES_FSVST'))
        self.assertEqual(record.dr_positive[481], ('P07332', 'FES_HUMAN'))
        self.assertEqual(record.dr_positive[482], ('P16879', 'FES_MOUSE'))
        self.assertEqual(record.dr_positive[483], ('P21804', 'FGFR1_CHICK'))
        self.assertEqual(record.dr_positive[484], ('Q07407', 'FGFR1_DROME'))
        self.assertEqual(record.dr_positive[485], ('P11362', 'FGFR1_HUMAN'))
        self.assertEqual(record.dr_positive[486], ('P16092', 'FGFR1_MOUSE'))
        self.assertEqual(record.dr_positive[487], ('Q04589', 'FGFR1_RAT'))
        self.assertEqual(record.dr_positive[488], ('P22182', 'FGFR1_XENLA'))
        self.assertEqual(record.dr_positive[489], ('Q09147', 'FGFR2_DROME'))
        self.assertEqual(record.dr_positive[490], ('P21802', 'FGFR2_HUMAN'))
        self.assertEqual(record.dr_positive[491], ('P21803', 'FGFR2_MOUSE'))
        self.assertEqual(record.dr_positive[492], ('Q03364', 'FGFR2_XENLA'))
        self.assertEqual(record.dr_positive[493], ('P22607', 'FGFR3_HUMAN'))
        self.assertEqual(record.dr_positive[494], ('Q61851', 'FGFR3_MOUSE'))
        self.assertEqual(record.dr_positive[495], ('P22455', 'FGFR4_HUMAN'))
        self.assertEqual(record.dr_positive[496], ('Q03142', 'FGFR4_MOUSE'))
        self.assertEqual(record.dr_positive[497], ('Q498D6', 'FGFR4_RAT'))
        self.assertEqual(record.dr_positive[498], ('P00544', 'FGR_FSVGR'))
        self.assertEqual(record.dr_positive[499], ('P09769', 'FGR_HUMAN'))

    def read1_positive2(self, record):
        self.assertEqual(len(record.dr_positive), 1689)
        self.assertEqual(record.dr_positive[500], ('P14234', 'FGR_MOUSE'))
        self.assertEqual(record.dr_positive[501], ('P09760', 'FLK_RAT'))
        self.assertEqual(record.dr_positive[502], ('Q9NLA1', 'FLR4_CAEEL'))
        self.assertEqual(record.dr_positive[503], ('P36888', 'FLT3_HUMAN'))
        self.assertEqual(record.dr_positive[504], ('Q00342', 'FLT3_MOUSE'))
        self.assertEqual(record.dr_positive[505], ('P00541', 'FPS_AVISP'))
        self.assertEqual(record.dr_positive[506], ('P18106', 'FPS_DROME'))
        self.assertEqual(record.dr_positive[507], ('P00530', 'FPS_FUJSV'))
        self.assertEqual(record.dr_positive[508], ('P42685', 'FRK_HUMAN'))
        self.assertEqual(record.dr_positive[509], ('P16892', 'FUS3_YEAST'))
        self.assertEqual(record.dr_positive[510], ('P23647', 'FUSED_DROME'))
        self.assertEqual(record.dr_positive[511], ('Q99078', 'FUZ7_USTMA'))
        self.assertEqual(record.dr_positive[512], ('Q05876', 'FYN_CHICK'))
        self.assertEqual(record.dr_positive[513], ('P06241', 'FYN_HUMAN'))
        self.assertEqual(record.dr_positive[514], ('P39688', 'FYN_MOUSE'))
        self.assertEqual(record.dr_positive[515], ('P13406', 'FYN_XENLA'))
        self.assertEqual(record.dr_positive[516], ('P27446', 'FYN_XIPHE'))
        self.assertEqual(record.dr_positive[517], ('Q9P7J8', 'GAD8_SCHPO'))
        self.assertEqual(record.dr_positive[518], ('Q9LX30', 'GCN2_ARATH'))
        self.assertEqual(record.dr_positive[519], ('Q9HGN1', 'GCN2_SCHPO'))
        self.assertEqual(record.dr_positive[520], ('P15442', 'GCN2_YEAST'))
        self.assertEqual(record.dr_positive[521], ('P16788', 'GCVK_HCMVA'))
        self.assertEqual(record.dr_positive[522], ('Q68101', 'GCVK_HCMVT'))
        self.assertEqual(record.dr_positive[523], ('Q12263', 'GIN4_YEAST'))
        self.assertEqual(record.dr_positive[524], ('P32865', 'GPRK1_DROME'))
        self.assertEqual(record.dr_positive[525], ('P32866', 'GPRK2_DROME'))
        self.assertEqual(record.dr_positive[526], ('Q622Z7', 'GRK1_CAEBR'))
        self.assertEqual(record.dr_positive[527], ('Q09537', 'GRK1_CAEEL'))
        self.assertEqual(record.dr_positive[528], ('Q09639', 'GRK2_CAEEL'))
        self.assertEqual(record.dr_positive[529], ('P32298', 'GRK4_HUMAN'))
        self.assertEqual(record.dr_positive[530], ('O70291', 'GRK4_MOUSE'))
        self.assertEqual(record.dr_positive[531], ('P70507', 'GRK4_RAT'))
        self.assertEqual(record.dr_positive[532], ('P43249', 'GRK5_BOVIN'))
        self.assertEqual(record.dr_positive[533], ('P34947', 'GRK5_HUMAN'))
        self.assertEqual(record.dr_positive[534], ('Q8VEB1', 'GRK5_MOUSE'))
        self.assertEqual(record.dr_positive[535], ('Q62833', 'GRK5_RAT'))
        self.assertEqual(record.dr_positive[536], ('P43250', 'GRK6_HUMAN'))
        self.assertEqual(record.dr_positive[537], ('O70293', 'GRK6_MOUSE'))
        self.assertEqual(record.dr_positive[538], ('P97711', 'GRK6_RAT'))
        self.assertEqual(record.dr_positive[539], ('Q8WTQ7', 'GRK7_HUMAN'))
        self.assertEqual(record.dr_positive[540], ('Q8WP15', 'GRK7_PIG'))
        self.assertEqual(record.dr_positive[541], ('Q9Z2G7', 'GRK7_SPETR'))
        self.assertEqual(record.dr_positive[542], ('Q9URT9', 'GSK31_SCHPO'))
        self.assertEqual(record.dr_positive[543], ('P49840', 'GSK3A_HUMAN'))
        self.assertEqual(record.dr_positive[544], ('P18265', 'GSK3A_RAT'))
        self.assertEqual(record.dr_positive[545], ('P49841', 'GSK3B_HUMAN'))
        self.assertEqual(record.dr_positive[546], ('Q9WV60', 'GSK3B_MOUSE'))
        self.assertEqual(record.dr_positive[547], ('P18266', 'GSK3B_RAT'))
        self.assertEqual(record.dr_positive[548], ('P51136', 'GSK3H_DICDI'))
        self.assertEqual(record.dr_positive[549], ('P83101', 'GSK3H_DROME'))
        self.assertEqual(record.dr_positive[550], ('Q10452', 'GSK3_SCHPO'))
        self.assertEqual(record.dr_positive[551], ('O14019', 'HAL4_SCHPO'))
        self.assertEqual(record.dr_positive[552], ('P25333', 'HAL4_YEAST'))
        self.assertEqual(record.dr_positive[553], ('P83103', 'HASP_DROME'))
        self.assertEqual(record.dr_positive[554], ('Q8TF76', 'HASP_HUMAN'))
        self.assertEqual(record.dr_positive[555], ('Q9Z0R0', 'HASP_MOUSE'))
        self.assertEqual(record.dr_positive[556], ('P08631', 'HCK_HUMAN'))
        self.assertEqual(record.dr_positive[557], ('Q95M30', 'HCK_MACFA'))
        self.assertEqual(record.dr_positive[558], ('P08103', 'HCK_MOUSE'))
        self.assertEqual(record.dr_positive[559], ('P50545', 'HCK_RAT'))
        self.assertEqual(record.dr_positive[560], ('Q23977', 'HEP_DROME'))
        self.assertEqual(record.dr_positive[561], ('P40235', 'HHP1_SCHPO'))
        self.assertEqual(record.dr_positive[562], ('P40236', 'HHP2_SCHPO'))
        self.assertEqual(record.dr_positive[563], ('Q86Z02', 'HIPK1_HUMAN'))
        self.assertEqual(record.dr_positive[564], ('O88904', 'HIPK1_MOUSE'))
        self.assertEqual(record.dr_positive[565], ('Q9H2X6', 'HIPK2_HUMAN'))
        self.assertEqual(record.dr_positive[566], ('Q9WUM7', 'HIPK2_MESAU'))
        self.assertEqual(record.dr_positive[567], ('Q9QZR5', 'HIPK2_MOUSE'))
        self.assertEqual(record.dr_positive[568], ('Q9H422', 'HIPK3_HUMAN'))
        self.assertEqual(record.dr_positive[569], ('Q9ERH7', 'HIPK3_MOUSE'))
        self.assertEqual(record.dr_positive[570], ('O88850', 'HIPK3_RAT'))
        self.assertEqual(record.dr_positive[571], ('Q8T0S6', 'HIPPO_DROME'))
        self.assertEqual(record.dr_positive[572], ('Q750A9', 'HOG1_ASHGO'))
        self.assertEqual(record.dr_positive[573], ('Q92207', 'HOG1_CANAL'))
        self.assertEqual(record.dr_positive[574], ('Q6FIU2', 'HOG1_CANGA'))
        self.assertEqual(record.dr_positive[575], ('Q9UV50', 'HOG1_DEBHA'))
        self.assertEqual(record.dr_positive[576], ('P32485', 'HOG1_YEAST'))
        self.assertEqual(record.dr_positive[577], ('O93982', 'HOG1_ZYGRO'))
        self.assertEqual(record.dr_positive[578], ('Q08732', 'HRK1_YEAST'))
        self.assertEqual(record.dr_positive[579], ('P29295', 'HRR25_YEAST'))
        self.assertEqual(record.dr_positive[580], ('P53356', 'HTK16_HYDAT'))
        self.assertEqual(record.dr_positive[581], ('Q25197', 'HTK7_HYDAT'))
        self.assertEqual(record.dr_positive[582], ('P57058', 'HUNK_HUMAN'))
        self.assertEqual(record.dr_positive[583], ('O88866', 'HUNK_MOUSE'))
        self.assertEqual(record.dr_positive[584], ('Q68UT7', 'HUNK_PANTR'))
        self.assertEqual(record.dr_positive[585], ('Q9UPZ9', 'ICK_HUMAN'))
        self.assertEqual(record.dr_positive[586], ('Q9JKV2', 'ICK_MOUSE'))
        self.assertEqual(record.dr_positive[587], ('Q62726', 'ICK_RAT'))
        self.assertEqual(record.dr_positive[588], ('Q6CWQ4', 'ICL1_KLULA'))
        self.assertEqual(record.dr_positive[589], ('Q05688', 'IGF1R_BOVIN'))
        self.assertEqual(record.dr_positive[590], ('P08069', 'IGF1R_HUMAN'))
        self.assertEqual(record.dr_positive[591], ('Q60751', 'IGF1R_MOUSE'))
        self.assertEqual(record.dr_positive[592], ('P24062', 'IGF1R_RAT'))
        self.assertEqual(record.dr_positive[593], ('O73798', 'IGF1R_XENLA'))
        self.assertEqual(record.dr_positive[594], ('O15111', 'IKKA_HUMAN'))
        self.assertEqual(record.dr_positive[595], ('Q60680', 'IKKA_MOUSE'))
        self.assertEqual(record.dr_positive[596], ('Q14164', 'IKKE_HUMAN'))
        self.assertEqual(record.dr_positive[597], ('Q9R0T8', 'IKKE_MOUSE'))
        self.assertEqual(record.dr_positive[598], ('O02466', 'ILPR_BRALA'))
        self.assertEqual(record.dr_positive[599], ('P14617', 'INSRR_CAVPO'))
        self.assertEqual(record.dr_positive[600], ('P14616', 'INSRR_HUMAN'))
        self.assertEqual(record.dr_positive[601], ('Q9WTL4', 'INSRR_MOUSE'))
        self.assertEqual(record.dr_positive[602], ('Q93105', 'INSR_AEDAE'))
        self.assertEqual(record.dr_positive[603], ('P09208', 'INSR_DROME'))
        self.assertEqual(record.dr_positive[604], ('P06213', 'INSR_HUMAN'))
        self.assertEqual(record.dr_positive[605], ('P15208', 'INSR_MOUSE'))
        self.assertEqual(record.dr_positive[606], ('P15127', 'INSR_RAT'))
        self.assertEqual(record.dr_positive[607], ('Q9PVZ4', 'INSR_XENLA'))
        self.assertEqual(record.dr_positive[608], ('Q755C4', 'IPL1_ASHGO'))
        self.assertEqual(record.dr_positive[609], ('Q6FV07', 'IPL1_CANGA'))
        self.assertEqual(record.dr_positive[610], ('Q6C3J2', 'IPL1_YARLI'))
        self.assertEqual(record.dr_positive[611], ('P38991', 'IPL1_YEAST'))
        self.assertEqual(record.dr_positive[612], ('P51617', 'IRAK1_HUMAN'))
        self.assertEqual(record.dr_positive[613], ('Q62406', 'IRAK1_MOUSE'))
        self.assertEqual(record.dr_positive[614], ('Q06098', 'ISR1_YEAST'))
        self.assertEqual(record.dr_positive[615], ('Q08881', 'ITK_HUMAN'))
        self.assertEqual(record.dr_positive[616], ('Q03526', 'ITK_MOUSE'))
        self.assertEqual(record.dr_positive[617], ('O12990', 'JAK1_BRARE'))
        self.assertEqual(record.dr_positive[618], ('Q09178', 'JAK1_CYPCA'))
        self.assertEqual(record.dr_positive[619], ('P23458', 'JAK1_HUMAN'))
        self.assertEqual(record.dr_positive[620], ('P52332', 'JAK1_MOUSE'))
        self.assertEqual(record.dr_positive[621], ('O60674', 'JAK2_HUMAN'))
        self.assertEqual(record.dr_positive[622], ('Q62120', 'JAK2_MOUSE'))
        self.assertEqual(record.dr_positive[623], ('Q62689', 'JAK2_RAT'))
        self.assertEqual(record.dr_positive[624], ('P52333', 'JAK3_HUMAN'))
        self.assertEqual(record.dr_positive[625], ('Q62137', 'JAK3_MOUSE'))
        self.assertEqual(record.dr_positive[626], ('Q63272', 'JAK3_RAT'))
        self.assertEqual(record.dr_positive[627], ('Q24592', 'JAK_DROME'))
        self.assertEqual(record.dr_positive[628], ('Q9V3I5', 'JIL1_DROME'))
        self.assertEqual(record.dr_positive[629], ('Q09792', 'KAA8_SCHPO'))
        self.assertEqual(record.dr_positive[630], ('Q09815', 'KAB7_SCHPO'))
        self.assertEqual(record.dr_positive[631], ('Q10078', 'KAND_SCHPO'))
        self.assertEqual(record.dr_positive[632], ('P06244', 'KAPA_YEAST'))
        self.assertEqual(record.dr_positive[633], ('P05131', 'KAPB1_BOVIN'))
        self.assertEqual(record.dr_positive[634], ('P24256', 'KAPB2_BOVIN'))
        self.assertEqual(record.dr_positive[635], ('P40376', 'KAPB_SCHPO'))
        self.assertEqual(record.dr_positive[636], ('P06245', 'KAPB_YEAST'))
        self.assertEqual(record.dr_positive[637], ('P00517', 'KAPCA_BOVIN'))
        self.assertEqual(record.dr_positive[638], ('Q8MJ44', 'KAPCA_CANFA'))
        self.assertEqual(record.dr_positive[639], ('P25321', 'KAPCA_CRIGR'))
        self.assertEqual(record.dr_positive[640], ('P17612', 'KAPCA_HUMAN'))
        self.assertEqual(record.dr_positive[641], ('P05132', 'KAPCA_MOUSE'))
        self.assertEqual(record.dr_positive[642], ('P36887', 'KAPCA_PIG'))
        self.assertEqual(record.dr_positive[643], ('P27791', 'KAPCA_RAT'))
        self.assertEqual(record.dr_positive[644], ('Q9MZD9', 'KAPCA_SHEEP'))
        self.assertEqual(record.dr_positive[645], ('P68180', 'KAPCB_CRIGR'))
        self.assertEqual(record.dr_positive[646], ('P22694', 'KAPCB_HUMAN'))
        self.assertEqual(record.dr_positive[647], ('P68181', 'KAPCB_MOUSE'))
        self.assertEqual(record.dr_positive[648], ('P05383', 'KAPCB_PIG'))
        self.assertEqual(record.dr_positive[649], ('P68182', 'KAPCB_RAT'))
        self.assertEqual(record.dr_positive[650], ('P22612', 'KAPCG_HUMAN'))
        self.assertEqual(record.dr_positive[651], ('O62846', 'KAPCG_MACMU'))
        self.assertEqual(record.dr_positive[652], ('P49673', 'KAPC_ASCSU'))
        self.assertEqual(record.dr_positive[653], ('P21137', 'KAPC_CAEEL'))
        self.assertEqual(record.dr_positive[654], ('P34099', 'KAPC_DICDI'))
        self.assertEqual(record.dr_positive[655], ('P12370', 'KAPC_DROME'))
        self.assertEqual(record.dr_positive[656], ('Q8SRK8', 'KAPC_ENCCU'))
        self.assertEqual(record.dr_positive[657], ('P05986', 'KAPC_YEAST'))
        self.assertEqual(record.dr_positive[658], ('P21901', 'KAPL_APLCA'))
        self.assertEqual(record.dr_positive[659], ('P38070', 'KBN8_YEAST'))
        self.assertEqual(record.dr_positive[660], ('P23291', 'KC11_YEAST'))
        self.assertEqual(record.dr_positive[661], ('P23292', 'KC12_YEAST'))
        self.assertEqual(record.dr_positive[662], ('Q8N752', 'KC1AL_HUMAN'))
        self.assertEqual(record.dr_positive[663], ('P67827', 'KC1A_BOVIN'))
        self.assertEqual(record.dr_positive[664], ('P67962', 'KC1A_CHICK'))
        self.assertEqual(record.dr_positive[665], ('P54367', 'KC1A_DROME'))
        self.assertEqual(record.dr_positive[666], ('P48729', 'KC1A_HUMAN'))
        self.assertEqual(record.dr_positive[667], ('Q8BK63', 'KC1A_MOUSE'))
        self.assertEqual(record.dr_positive[668], ('P67828', 'KC1A_RABIT'))
        self.assertEqual(record.dr_positive[669], ('P97633', 'KC1A_RAT'))
        self.assertEqual(record.dr_positive[670], ('P67829', 'KC1A_SHEEP'))
        self.assertEqual(record.dr_positive[671], ('P67963', 'KC1A_XENLA'))
        self.assertEqual(record.dr_positive[672], ('P35507', 'KC1B_BOVIN'))
        self.assertEqual(record.dr_positive[673], ('P42158', 'KC1D_ARATH'))
        self.assertEqual(record.dr_positive[674], ('P48730', 'KC1D_HUMAN'))
        self.assertEqual(record.dr_positive[675], ('Q9DC28', 'KC1D_MOUSE'))
        self.assertEqual(record.dr_positive[676], ('Q06486', 'KC1D_RAT'))
        self.assertEqual(record.dr_positive[677], ('P49674', 'KC1E_HUMAN'))
        self.assertEqual(record.dr_positive[678], ('Q9JMK2', 'KC1E_MOUSE'))
        self.assertEqual(record.dr_positive[679], ('Q9HCP0', 'KC1G1_HUMAN'))
        self.assertEqual(record.dr_positive[680], ('Q8BTH8', 'KC1G1_MOUSE'))
        self.assertEqual(record.dr_positive[681], ('Q62761', 'KC1G1_RAT'))
        self.assertEqual(record.dr_positive[682], ('P78368', 'KC1G2_HUMAN'))
        self.assertEqual(record.dr_positive[683], ('Q8BVP5', 'KC1G2_MOUSE'))
        self.assertEqual(record.dr_positive[684], ('Q62762', 'KC1G2_RAT'))
        self.assertEqual(record.dr_positive[685], ('P35509', 'KC1G3_BOVIN'))
        self.assertEqual(record.dr_positive[686], ('Q9Y6M4', 'KC1G3_HUMAN'))
        self.assertEqual(record.dr_positive[687], ('Q62763', 'KC1G3_RAT'))
        self.assertEqual(record.dr_positive[688], ('Q6QNL9', 'KC1_EIMTE'))
        self.assertEqual(record.dr_positive[689], ('O15726', 'KC1_PLAF4'))
        self.assertEqual(record.dr_positive[690], ('Q8IHZ9', 'KC1_PLAF7'))
        self.assertEqual(record.dr_positive[691], ('Q7RBX5', 'KC1_PLAYO'))
        self.assertEqual(record.dr_positive[692], ('Q6QNM1', 'KC1_TOXGO'))
        self.assertEqual(record.dr_positive[693], ('P29620', 'KC47_ORYSA'))
        self.assertEqual(record.dr_positive[694], ('Q14012', 'KCC1A_HUMAN'))
        self.assertEqual(record.dr_positive[695], ('Q91YS8', 'KCC1A_MOUSE'))
        self.assertEqual(record.dr_positive[696], ('Q63450', 'KCC1A_RAT'))
        self.assertEqual(record.dr_positive[697], ('Q6P2M8', 'KCC1B_HUMAN'))
        self.assertEqual(record.dr_positive[698], ('Q9QYK9', 'KCC1B_MOUSE'))
        self.assertEqual(record.dr_positive[699], ('O70150', 'KCC1B_RAT'))
        self.assertEqual(record.dr_positive[700], ('Q8IU85', 'KCC1D_HUMAN'))
        self.assertEqual(record.dr_positive[701], ('Q8BW96', 'KCC1D_MOUSE'))
        self.assertEqual(record.dr_positive[702], ('Q96NX5', 'KCC1G_HUMAN'))
        self.assertEqual(record.dr_positive[703], ('Q91VB2', 'KCC1G_MOUSE'))
        self.assertEqual(record.dr_positive[704], ('Q7TNJ7', 'KCC1G_RAT'))
        self.assertEqual(record.dr_positive[705], ('Q00771', 'KCC1_EMENI'))
        self.assertEqual(record.dr_positive[706], ('O14408', 'KCC1_METAN'))
        self.assertEqual(record.dr_positive[707], ('Q9P7I2', 'KCC1_SCHPO'))
        self.assertEqual(record.dr_positive[708], ('P27466', 'KCC1_YEAST'))
        self.assertEqual(record.dr_positive[709], ('Q00168', 'KCC2A_DROME'))
        self.assertEqual(record.dr_positive[710], ('Q9UQM7', 'KCC2A_HUMAN'))
        self.assertEqual(record.dr_positive[711], ('P11798', 'KCC2A_MOUSE'))
        self.assertEqual(record.dr_positive[712], ('Q5RCC4', 'KCC2A_PONPY'))
        self.assertEqual(record.dr_positive[713], ('P11275', 'KCC2A_RAT'))
        self.assertEqual(record.dr_positive[714], ('Q13554', 'KCC2B_HUMAN'))
        self.assertEqual(record.dr_positive[715], ('P28652', 'KCC2B_MOUSE'))
        self.assertEqual(record.dr_positive[716], ('P08413', 'KCC2B_RAT'))
        self.assertEqual(record.dr_positive[717], ('Q13557', 'KCC2D_HUMAN'))
        self.assertEqual(record.dr_positive[718], ('P15791', 'KCC2D_RAT'))
        self.assertEqual(record.dr_positive[719], ('Q13555', 'KCC2G_HUMAN'))
        self.assertEqual(record.dr_positive[720], ('Q923T9', 'KCC2G_MOUSE'))
        self.assertEqual(record.dr_positive[721], ('P11730', 'KCC2G_RAT'))
        self.assertEqual(record.dr_positive[722], ('O42844', 'KCC2_SCHPO'))
        self.assertEqual(record.dr_positive[723], ('P22517', 'KCC2_YEAST'))
        self.assertEqual(record.dr_positive[724], ('Q16566', 'KCC4_HUMAN'))
        self.assertEqual(record.dr_positive[725], ('P08414', 'KCC4_MOUSE'))
        self.assertEqual(record.dr_positive[726], ('P13234', 'KCC4_RAT'))
        self.assertEqual(record.dr_positive[727], ('P25389', 'KCC4_YEAST'))
        self.assertEqual(record.dr_positive[728], ('Q07250', 'KCCS_MALDO'))
        self.assertEqual(record.dr_positive[729], ('P16911', 'KDC1_DROME'))
        self.assertEqual(record.dr_positive[730], ('P16912', 'KDC2_DROME'))
        self.assertEqual(record.dr_positive[731], ('P00545', 'KFMS_FSVMD'))
        self.assertEqual(record.dr_positive[732], ('P00516', 'KGP1A_BOVIN'))
        self.assertEqual(record.dr_positive[733], ('Q13976', 'KGP1A_HUMAN'))
        self.assertEqual(record.dr_positive[734], ('O77676', 'KGP1A_RABIT'))
        self.assertEqual(record.dr_positive[735], ('P21136', 'KGP1B_BOVIN'))
        self.assertEqual(record.dr_positive[736], ('P14619', 'KGP1B_HUMAN'))
        self.assertEqual(record.dr_positive[737], ('Q9Z0Z0', 'KGP1B_MOUSE'))
        self.assertEqual(record.dr_positive[738], ('Q03042', 'KGP1_DROME'))
        self.assertEqual(record.dr_positive[739], ('Q03043', 'KGP24_DROME'))
        self.assertEqual(record.dr_positive[740], ('P32023', 'KGP25_DROME'))
        self.assertEqual(record.dr_positive[741], ('Q13237', 'KGP2_HUMAN'))
        self.assertEqual(record.dr_positive[742], ('Q61410', 'KGP2_MOUSE'))
        self.assertEqual(record.dr_positive[743], ('Q64595', 'KGP2_RAT'))
        self.assertEqual(record.dr_positive[744], ('P43637', 'KGS9_YEAST'))
        self.assertEqual(record.dr_positive[745], ('P38692', 'KIC1_YEAST'))
        self.assertEqual(record.dr_positive[746], ('Q38997', 'KIN10_ARATH'))
        self.assertEqual(record.dr_positive[747], ('P92958', 'KIN11_ARATH'))
        self.assertEqual(record.dr_positive[748], ('P34891', 'KIN15_CAEEL'))
        self.assertEqual(record.dr_positive[749], ('P34892', 'KIN16_CAEEL'))
        self.assertEqual(record.dr_positive[750], ('P22987', 'KIN1_SCHPO'))
        self.assertEqual(record.dr_positive[751], ('P13185', 'KIN1_YEAST'))
        self.assertEqual(record.dr_positive[752], ('Q20471', 'KIN20_CAEEL'))
        self.assertEqual(record.dr_positive[753], ('Q10925', 'KIN25_CAEEL'))
        self.assertEqual(record.dr_positive[754], ('P06242', 'KIN28_YEAST'))
        self.assertEqual(record.dr_positive[755], ('P13186', 'KIN2_YEAST'))
        self.assertEqual(record.dr_positive[756], ('P34265', 'KIN31_CAEEL'))
        self.assertEqual(record.dr_positive[757], ('P22209', 'KIN3_YEAST'))
        self.assertEqual(record.dr_positive[758], ('Q01919', 'KIN4_YEAST'))
        self.assertEqual(record.dr_positive[759], ('P43481', 'KIT_BOVIN'))
        self.assertEqual(record.dr_positive[760], ('O97799', 'KIT_CANFA'))
        self.assertEqual(record.dr_positive[761], ('Q28317', 'KIT_CAPHI'))
        self.assertEqual(record.dr_positive[762], ('Q08156', 'KIT_CHICK'))
        self.assertEqual(record.dr_positive[763], ('Q28889', 'KIT_FELCA'))
        self.assertEqual(record.dr_positive[764], ('P04048', 'KIT_FSVHZ'))
        self.assertEqual(record.dr_positive[765], ('P10721', 'KIT_HUMAN'))
        self.assertEqual(record.dr_positive[766], ('P05532', 'KIT_MOUSE'))
        self.assertEqual(record.dr_positive[767], ('Q9HFF4', 'KK31_SCHPO'))
        self.assertEqual(record.dr_positive[768], ('Q9P6P3', 'KKB3_SCHPO'))
        self.assertEqual(record.dr_positive[769], ('Q8N5S9', 'KKCC1_HUMAN'))
        self.assertEqual(record.dr_positive[770], ('Q8VBY2', 'KKCC1_MOUSE'))
        self.assertEqual(record.dr_positive[771], ('P97756', 'KKCC1_RAT'))
        self.assertEqual(record.dr_positive[772], ('Q96RR4', 'KKCC2_HUMAN'))
        self.assertEqual(record.dr_positive[773], ('Q8C078', 'KKCC2_MOUSE'))
        self.assertEqual(record.dr_positive[774], ('O88831', 'KKCC2_RAT'))
        self.assertEqual(record.dr_positive[775], ('Q9UTH3', 'KKE1_SCHPO'))
        self.assertEqual(record.dr_positive[776], ('P34244', 'KKK1_YEAST'))
        self.assertEqual(record.dr_positive[777], ('P28708', 'KKL6_YEAST'))
        self.assertEqual(record.dr_positive[778], ('P36005', 'KKQ1_YEAST'))
        self.assertEqual(record.dr_positive[779], ('P36003', 'KKR1_YEAST'))
        self.assertEqual(record.dr_positive[780], ('Q03533', 'KM8S_YEAST'))
        self.assertEqual(record.dr_positive[781], ('P32350', 'KNS1_YEAST'))
        self.assertEqual(record.dr_positive[782], ('Q9Y7J6', 'KOIA_SCHPO'))
        self.assertEqual(record.dr_positive[783], ('Q12236', 'KOK0_YEAST'))
        self.assertEqual(record.dr_positive[784], ('Q12222', 'KOM8_YEAST'))
        self.assertEqual(record.dr_positive[785], ('P34885', 'KPC1B_CAEEL'))
        self.assertEqual(record.dr_positive[786], ('Q16974', 'KPC1_APLCA'))
        self.assertEqual(record.dr_positive[787], ('Q00078', 'KPC1_ASPNG'))
        self.assertEqual(record.dr_positive[788], ('P34722', 'KPC1_CAEEL'))
        self.assertEqual(record.dr_positive[789], ('P43057', 'KPC1_CANAL'))
        self.assertEqual(record.dr_positive[790], ('O42632', 'KPC1_COCHE'))
        self.assertEqual(record.dr_positive[791], ('P05130', 'KPC1_DROME'))
        self.assertEqual(record.dr_positive[792], ('Q25378', 'KPC1_LYTPI'))
        self.assertEqual(record.dr_positive[793], ('P87253', 'KPC1_NEUCR'))
        self.assertEqual(record.dr_positive[794], ('Q99014', 'KPC1_TRIRE'))
        self.assertEqual(record.dr_positive[795], ('P24583', 'KPC1_YEAST'))
        self.assertEqual(record.dr_positive[796], ('Q16975', 'KPC2_APLCA'))
        self.assertEqual(record.dr_positive[797], ('P90980', 'KPC2_CAEEL'))
        self.assertEqual(record.dr_positive[798], ('P13677', 'KPC2_DROME'))
        self.assertEqual(record.dr_positive[799], ('Q19266', 'KPC3_CAEEL'))
        self.assertEqual(record.dr_positive[800], ('P13678', 'KPC3_DROME'))
        self.assertEqual(record.dr_positive[801], ('P83099', 'KPC4_DROME'))
        self.assertEqual(record.dr_positive[802], ('P04409', 'KPCA_BOVIN'))
        self.assertEqual(record.dr_positive[803], ('P17252', 'KPCA_HUMAN'))
        self.assertEqual(record.dr_positive[804], ('P20444', 'KPCA_MOUSE'))
        self.assertEqual(record.dr_positive[805], ('P10102', 'KPCA_RABIT'))
        self.assertEqual(record.dr_positive[806], ('P05696', 'KPCA_RAT'))
        self.assertEqual(record.dr_positive[807], ('P05126', 'KPCB_BOVIN'))
        self.assertEqual(record.dr_positive[808], ('P05771', 'KPCB_HUMAN'))
        self.assertEqual(record.dr_positive[809], ('P68404', 'KPCB_MOUSE'))
        self.assertEqual(record.dr_positive[810], ('P05772', 'KPCB_RABIT'))
        self.assertEqual(record.dr_positive[811], ('P68403', 'KPCB_RAT'))
        self.assertEqual(record.dr_positive[812], ('Q15139', 'KPCD1_HUMAN'))
        self.assertEqual(record.dr_positive[813], ('Q62101', 'KPCD1_MOUSE'))
        self.assertEqual(record.dr_positive[814], ('Q9BZL6', 'KPCD2_HUMAN'))
        self.assertEqual(record.dr_positive[815], ('O94806', 'KPCD3_HUMAN'))
        self.assertEqual(record.dr_positive[816], ('Q8K1Y2', 'KPCD3_MOUSE'))
        self.assertEqual(record.dr_positive[817], ('Q5PU49', 'KPCD_CANFA'))
        self.assertEqual(record.dr_positive[818], ('Q05655', 'KPCD_HUMAN'))
        self.assertEqual(record.dr_positive[819], ('P28867', 'KPCD_MOUSE'))
        self.assertEqual(record.dr_positive[820], ('P09215', 'KPCD_RAT'))
        self.assertEqual(record.dr_positive[821], ('Q02156', 'KPCE_HUMAN'))
        self.assertEqual(record.dr_positive[822], ('P16054', 'KPCE_MOUSE'))
        self.assertEqual(record.dr_positive[823], ('P10830', 'KPCE_RABIT'))
        self.assertEqual(record.dr_positive[824], ('P09216', 'KPCE_RAT'))
        self.assertEqual(record.dr_positive[825], ('P05128', 'KPCG_BOVIN'))
        self.assertEqual(record.dr_positive[826], ('P05129', 'KPCG_HUMAN'))
        self.assertEqual(record.dr_positive[827], ('P63318', 'KPCG_MOUSE'))
        self.assertEqual(record.dr_positive[828], ('P10829', 'KPCG_RABIT'))
        self.assertEqual(record.dr_positive[829], ('P63319', 'KPCG_RAT'))
        self.assertEqual(record.dr_positive[830], ('Q90XF2', 'KPCI_BRARE'))
        self.assertEqual(record.dr_positive[831], ('P41743', 'KPCI_HUMAN'))
        self.assertEqual(record.dr_positive[832], ('Q62074', 'KPCI_MOUSE'))
        self.assertEqual(record.dr_positive[833], ('Q5R4K9', 'KPCI_PONPY'))
        self.assertEqual(record.dr_positive[834], ('P24723', 'KPCL_HUMAN'))
        self.assertEqual(record.dr_positive[835], ('P23298', 'KPCL_MOUSE'))
        self.assertEqual(record.dr_positive[836], ('Q64617', 'KPCL_RAT'))
        self.assertEqual(record.dr_positive[837], ('Q04759', 'KPCT_HUMAN'))
        self.assertEqual(record.dr_positive[838], ('Q02111', 'KPCT_MOUSE'))
        self.assertEqual(record.dr_positive[839], ('Q05513', 'KPCZ_HUMAN'))
        self.assertEqual(record.dr_positive[840], ('Q02956', 'KPCZ_MOUSE'))
        self.assertEqual(record.dr_positive[841], ('O19111', 'KPCZ_RABIT'))
        self.assertEqual(record.dr_positive[842], ('P09217', 'KPCZ_RAT'))
        self.assertEqual(record.dr_positive[843], ('Q05652', 'KPEL_DROME'))
        self.assertEqual(record.dr_positive[844], ('Q39030', 'KPK19_ARATH'))
        self.assertEqual(record.dr_positive[845], ('P42818', 'KPK1_ARATH'))
        self.assertEqual(record.dr_positive[846], ('Q02595', 'KPK2_PLAFK'))
        self.assertEqual(record.dr_positive[847], ('P17801', 'KPRO_MAIZE'))
        self.assertEqual(record.dr_positive[848], ('Q501V0', 'KPSH1_BRARE'))
        self.assertEqual(record.dr_positive[849], ('P11801', 'KPSH1_HUMAN'))
        self.assertEqual(record.dr_positive[850], ('Q91YA2', 'KPSH1_MOUSE'))
        self.assertEqual(record.dr_positive[851], ('Q4KTY1', 'KPSH1_PINFU'))
        self.assertEqual(record.dr_positive[852], ('Q96QS6', 'KPSH2_HUMAN'))
        self.assertEqual(record.dr_positive[853], ('P54644', 'KRAC_DICDI'))
        self.assertEqual(record.dr_positive[854], ('Q07292', 'KRAF1_CAEEL'))
        self.assertEqual(record.dr_positive[855], ('P11346', 'KRAF1_DROME'))
        self.assertEqual(record.dr_positive[856], ('Q15418', 'KS6A1_HUMAN'))
        self.assertEqual(record.dr_positive[857], ('P18653', 'KS6A1_MOUSE'))
        self.assertEqual(record.dr_positive[858], ('Q63531', 'KS6A1_RAT'))
        self.assertEqual(record.dr_positive[859], ('Q15349', 'KS6A2_HUMAN'))
        self.assertEqual(record.dr_positive[860], ('Q9WUT3', 'KS6A2_MOUSE'))
        self.assertEqual(record.dr_positive[861], ('P51812', 'KS6A3_HUMAN'))
        self.assertEqual(record.dr_positive[862], ('P18654', 'KS6A3_MOUSE'))
        self.assertEqual(record.dr_positive[863], ('O75676', 'KS6A4_HUMAN'))
        self.assertEqual(record.dr_positive[864], ('Q9Z2B9', 'KS6A4_MOUSE'))
        self.assertEqual(record.dr_positive[865], ('O75582', 'KS6A5_HUMAN'))
        self.assertEqual(record.dr_positive[866], ('Q8C050', 'KS6A5_MOUSE'))
        self.assertEqual(record.dr_positive[867], ('Q9UK32', 'KS6A6_HUMAN'))
        self.assertEqual(record.dr_positive[868], ('P18652', 'KS6AA_CHICK'))
        self.assertEqual(record.dr_positive[869], ('P10665', 'KS6AA_XENLA'))
        self.assertEqual(record.dr_positive[870], ('P10666', 'KS6AB_XENLA'))
        self.assertEqual(record.dr_positive[871], ('Q21734', 'KS6A_CAEEL'))
        self.assertEqual(record.dr_positive[872], ('P23443', 'KS6B1_HUMAN'))
        self.assertEqual(record.dr_positive[873], ('Q8BSK8', 'KS6B1_MOUSE'))
        self.assertEqual(record.dr_positive[874], ('P67998', 'KS6B1_RABIT'))
        self.assertEqual(record.dr_positive[875], ('P67999', 'KS6B1_RAT'))
        self.assertEqual(record.dr_positive[876], ('Q9UBS0', 'KS6B2_HUMAN'))
        self.assertEqual(record.dr_positive[877], ('Q9Z1M4', 'KS6B2_MOUSE'))
        self.assertEqual(record.dr_positive[878], ('Q39019', 'KSG10_ARATH'))
        self.assertEqual(record.dr_positive[879], ('P43288', 'KSG1_ARATH'))
        self.assertEqual(record.dr_positive[880], ('Q12701', 'KSG1_SCHPO'))
        self.assertEqual(record.dr_positive[881], ('O23145', 'KSG2_ARATH'))
        self.assertEqual(record.dr_positive[882], ('P43289', 'KSG3_ARATH'))
        self.assertEqual(record.dr_positive[883], ('Q9FVS6', 'KSG4_ARATH'))
        self.assertEqual(record.dr_positive[884], ('Q8VZD5', 'KSG5_ARATH'))
        self.assertEqual(record.dr_positive[885], ('Q39010', 'KSG6_ARATH'))
        self.assertEqual(record.dr_positive[886], ('Q39011', 'KSG7_ARATH'))
        self.assertEqual(record.dr_positive[887], ('Q96287', 'KSG8_ARATH'))
        self.assertEqual(record.dr_positive[888], ('Q39012', 'KSG9_ARATH'))
        self.assertEqual(record.dr_positive[889], ('O04160', 'KSGT_BRANA'))
        self.assertEqual(record.dr_positive[890], ('P14681', 'KSS1_YEAST'))
        self.assertEqual(record.dr_positive[891], ('P43405', 'KSYK_HUMAN'))
        self.assertEqual(record.dr_positive[892], ('P48025', 'KSYK_MOUSE'))
        self.assertEqual(record.dr_positive[893], ('Q00655', 'KSYK_PIG'))
        self.assertEqual(record.dr_positive[894], ('Q64725', 'KSYK_RAT'))
        self.assertEqual(record.dr_positive[895], ('P18160', 'KYK1_DICDI'))
        self.assertEqual(record.dr_positive[896], ('P18161', 'KYK2_DICDI'))
        self.assertEqual(record.dr_positive[897], ('Q9NRM7', 'LATS2_HUMAN'))
        self.assertEqual(record.dr_positive[898], ('Q7TSJ6', 'LATS2_MOUSE'))
        self.assertEqual(record.dr_positive[899], ('Q5PXS1', 'LCK_AOTNA'))
        self.assertEqual(record.dr_positive[900], ('P42683', 'LCK_CHICK'))
        self.assertEqual(record.dr_positive[901], ('P06239', 'LCK_HUMAN'))
        self.assertEqual(record.dr_positive[902], ('P06240', 'LCK_MOUSE'))
        self.assertEqual(record.dr_positive[903], ('Q95KR7', 'LCK_SAISC'))
        self.assertEqual(record.dr_positive[904], ('P24348', 'LET23_CAEEL'))
        self.assertEqual(record.dr_positive[905], ('Q8QFP8', 'LIMK1_CHICK'))
        self.assertEqual(record.dr_positive[906], ('Q8IR79', 'LIMK1_DROME'))
        self.assertEqual(record.dr_positive[907], ('P53667', 'LIMK1_HUMAN'))
        self.assertEqual(record.dr_positive[908], ('P53668', 'LIMK1_MOUSE'))
        self.assertEqual(record.dr_positive[909], ('P53669', 'LIMK1_RAT'))
        self.assertEqual(record.dr_positive[910], ('O42565', 'LIMK1_XENLA'))
        self.assertEqual(record.dr_positive[911], ('P53666', 'LIMK2_CHICK'))
        self.assertEqual(record.dr_positive[912], ('P53671', 'LIMK2_HUMAN'))
        self.assertEqual(record.dr_positive[913], ('O54785', 'LIMK2_MOUSE'))
        self.assertEqual(record.dr_positive[914], ('P53670', 'LIMK2_RAT'))
        self.assertEqual(record.dr_positive[915], ('Q10156', 'LKH1_SCHPO'))
        self.assertEqual(record.dr_positive[916], ('O61267', 'LOK_DROME'))
        self.assertEqual(record.dr_positive[917], ('Q5S007', 'LRRK2_HUMAN'))
        self.assertEqual(record.dr_positive[918], ('Q5S006', 'LRRK2_MOUSE'))
        self.assertEqual(record.dr_positive[919], ('P29376', 'LTK_HUMAN'))
        self.assertEqual(record.dr_positive[920], ('P08923', 'LTK_MOUSE'))
        self.assertEqual(record.dr_positive[921], ('P07948', 'LYN_HUMAN'))
        self.assertEqual(record.dr_positive[922], ('P25911', 'LYN_MOUSE'))
        self.assertEqual(record.dr_positive[923], ('Q07014', 'LYN_RAT'))
        self.assertEqual(record.dr_positive[924], ('Q02779', 'M3K10_HUMAN'))
        self.assertEqual(record.dr_positive[925], ('Q16584', 'M3K11_HUMAN'))
        self.assertEqual(record.dr_positive[926], ('Q99558', 'M3K14_HUMAN'))
        self.assertEqual(record.dr_positive[927], ('Q9WUL6', 'M3K14_MOUSE'))
        self.assertEqual(record.dr_positive[928], ('O22040', 'M3K1_ARATH'))
        self.assertEqual(record.dr_positive[929], ('Q13233', 'M3K1_HUMAN'))
        self.assertEqual(record.dr_positive[930], ('P53349', 'M3K1_MOUSE'))
        self.assertEqual(record.dr_positive[931], ('Q62925', 'M3K1_RAT'))
        self.assertEqual(record.dr_positive[932], ('Q9FZ36', 'M3K2_ARATH'))
        self.assertEqual(record.dr_positive[933], ('Q9Y2U5', 'M3K2_HUMAN'))
        self.assertEqual(record.dr_positive[934], ('Q61083', 'M3K2_MOUSE'))
        self.assertEqual(record.dr_positive[935], ('O22042', 'M3K3_ARATH'))
        self.assertEqual(record.dr_positive[936], ('Q9Y6R4', 'M3K4_HUMAN'))
        self.assertEqual(record.dr_positive[937], ('O08648', 'M3K4_MOUSE'))
        self.assertEqual(record.dr_positive[938], ('Q99683', 'M3K5_HUMAN'))
        self.assertEqual(record.dr_positive[939], ('O35099', 'M3K5_MOUSE'))
        self.assertEqual(record.dr_positive[940], ('O95382', 'M3K6_HUMAN'))
        self.assertEqual(record.dr_positive[941], ('Q9V3Q6', 'M3K7_DROME'))
        self.assertEqual(record.dr_positive[942], ('O43318', 'M3K7_HUMAN'))
        self.assertEqual(record.dr_positive[943], ('Q62073', 'M3K7_MOUSE'))
        self.assertEqual(record.dr_positive[944], ('P80192', 'M3K9_HUMAN'))
        self.assertEqual(record.dr_positive[945], ('Q5TCX8', 'M3KL4_HUMAN'))
        self.assertEqual(record.dr_positive[946], ('P83104', 'M3LK7_DROME'))
        self.assertEqual(record.dr_positive[947], ('Q92918', 'M4K1_HUMAN'))
        self.assertEqual(record.dr_positive[948], ('P70218', 'M4K1_MOUSE'))
        self.assertEqual(record.dr_positive[949], ('Q12851', 'M4K2_HUMAN'))
        self.assertEqual(record.dr_positive[950], ('Q61161', 'M4K2_MOUSE'))
        self.assertEqual(record.dr_positive[951], ('Q8IVH8', 'M4K3_HUMAN'))
        self.assertEqual(record.dr_positive[952], ('Q99JP0', 'M4K3_MOUSE'))
        self.assertEqual(record.dr_positive[953], ('Q924I2', 'M4K3_RAT'))
        self.assertEqual(record.dr_positive[954], ('O95819', 'M4K4_HUMAN'))
        self.assertEqual(record.dr_positive[955], ('P97820', 'M4K4_MOUSE'))
        self.assertEqual(record.dr_positive[956], ('Q9Y4K4', 'M4K5_HUMAN'))
        self.assertEqual(record.dr_positive[957], ('Q8BPM2', 'M4K5_MOUSE'))
        self.assertEqual(record.dr_positive[958], ('P20794', 'MAK_HUMAN'))
        self.assertEqual(record.dr_positive[959], ('Q04859', 'MAK_MOUSE'))
        self.assertEqual(record.dr_positive[960], ('P20793', 'MAK_RAT'))
        self.assertEqual(record.dr_positive[961], ('P49071', 'MAPK2_DROME'))
        self.assertEqual(record.dr_positive[962], ('P49137', 'MAPK2_HUMAN'))
        self.assertEqual(record.dr_positive[963], ('P49138', 'MAPK2_MOUSE'))
        self.assertEqual(record.dr_positive[964], ('P49139', 'MAPK2_RABIT'))
        self.assertEqual(record.dr_positive[965], ('Q3SYZ2', 'MAPK3_BOVIN'))
        self.assertEqual(record.dr_positive[966], ('Q16644', 'MAPK3_HUMAN'))
        self.assertEqual(record.dr_positive[967], ('Q3UMW7', 'MAPK3_MOUSE'))
        self.assertEqual(record.dr_positive[968], ('Q66H84', 'MAPK3_RAT'))
        self.assertEqual(record.dr_positive[969], ('Q00859', 'MAPK_FUSSO'))
        self.assertEqual(record.dr_positive[970], ('Q06060', 'MAPK_PEA'))
        self.assertEqual(record.dr_positive[971], ('Q40884', 'MAPK_PETHY'))
        self.assertEqual(record.dr_positive[972], ('Q9P0L2', 'MARK1_HUMAN'))
        self.assertEqual(record.dr_positive[973], ('Q8VHJ5', 'MARK1_MOUSE'))
        self.assertEqual(record.dr_positive[974], ('O08678', 'MARK1_RAT'))
        self.assertEqual(record.dr_positive[975], ('Q7KZI7', 'MARK2_HUMAN'))
        self.assertEqual(record.dr_positive[976], ('Q05512', 'MARK2_MOUSE'))
        self.assertEqual(record.dr_positive[977], ('O08679', 'MARK2_RAT'))
        self.assertEqual(record.dr_positive[978], ('P27448', 'MARK3_HUMAN'))
        self.assertEqual(record.dr_positive[979], ('Q03141', 'MARK3_MOUSE'))
        self.assertEqual(record.dr_positive[980], ('Q8VHF0', 'MARK3_RAT'))
        self.assertEqual(record.dr_positive[981], ('Q96L34', 'MARK4_HUMAN'))
        self.assertEqual(record.dr_positive[982], ('Q8CIP4', 'MARK4_MOUSE'))
        self.assertEqual(record.dr_positive[983], ('P42679', 'MATK_HUMAN'))
        self.assertEqual(record.dr_positive[984], ('P41242', 'MATK_MOUSE'))
        self.assertEqual(record.dr_positive[985], ('P41243', 'MATK_RAT'))
        self.assertEqual(record.dr_positive[986], ('P21965', 'MCK1_YEAST'))
        self.assertEqual(record.dr_positive[987], ('O43077', 'MDE3_SCHPO'))
        self.assertEqual(record.dr_positive[988], ('P38615', 'MDS1_YEAST'))
        self.assertEqual(record.dr_positive[989], ('Q10292', 'MEK1_SCHPO'))
        self.assertEqual(record.dr_positive[990], ('P24719', 'MEK1_YEAST'))
        self.assertEqual(record.dr_positive[991], ('Q10664', 'MEK2_CAEEL'))
        self.assertEqual(record.dr_positive[992], ('Q14680', 'MELK_HUMAN'))
        self.assertEqual(record.dr_positive[993], ('Q61846', 'MELK_MOUSE'))
        self.assertEqual(record.dr_positive[994], ('Q12866', 'MERTK_HUMAN'))
        self.assertEqual(record.dr_positive[995], ('Q60805', 'MERTK_MOUSE'))
        self.assertEqual(record.dr_positive[996], ('P57097', 'MERTK_RAT'))
        self.assertEqual(record.dr_positive[997], ('Q2QLG5', 'MET_CALJA'))
        self.assertEqual(record.dr_positive[998], ('Q2QLC0', 'MET_CALMO'))
        self.assertEqual(record.dr_positive[999], ('Q75ZY9', 'MET_CANFA'))

    def read1_positive3(self, record):
        self.assertEqual(len(record.dr_positive), 1689)
        self.assertEqual(record.dr_positive[1000], ('Q2QLA9', 'MET_HORSE'))
        self.assertEqual(record.dr_positive[1001], ('P08581', 'MET_HUMAN'))
        self.assertEqual(record.dr_positive[1002], ('Q2QL89', 'MET_MICMU'))
        self.assertEqual(record.dr_positive[1003], ('P16056', 'MET_MOUSE'))
        self.assertEqual(record.dr_positive[1004], ('Q2QLH6', 'MET_OTOGA'))
        self.assertEqual(record.dr_positive[1005], ('Q2QLF1', 'MET_PANTR'))
        self.assertEqual(record.dr_positive[1006], ('Q2QLE0', 'MET_PIG'))
        self.assertEqual(record.dr_positive[1007], ('P97523', 'MET_RAT'))
        self.assertEqual(record.dr_positive[1008], ('Q23356', 'MIG15_CAEEL'))
        self.assertEqual(record.dr_positive[1009], ('P00531', 'MIL_AVIMH'))
        self.assertEqual(record.dr_positive[1010], ('Q8N4C8', 'MINK1_HUMAN'))
        self.assertEqual(record.dr_positive[1011], ('Q9JM52', 'MINK1_MOUSE'))
        self.assertEqual(record.dr_positive[1012], ('Q25410', 'MIPR_LYMST'))
        self.assertEqual(record.dr_positive[1013], ('P46196', 'MK01_BOVIN'))
        self.assertEqual(record.dr_positive[1014], ('P28482', 'MK01_HUMAN'))
        self.assertEqual(record.dr_positive[1015], ('P63085', 'MK01_MOUSE'))
        self.assertEqual(record.dr_positive[1016], ('P63086', 'MK01_RAT'))
        self.assertEqual(record.dr_positive[1017], ('P26696', 'MK01_XENLA'))
        self.assertEqual(record.dr_positive[1018], ('P27361', 'MK03_HUMAN'))
        self.assertEqual(record.dr_positive[1019], ('Q63844', 'MK03_MOUSE'))
        self.assertEqual(record.dr_positive[1020], ('P21708', 'MK03_RAT'))
        self.assertEqual(record.dr_positive[1021], ('P31152', 'MK04_HUMAN'))
        self.assertEqual(record.dr_positive[1022], ('Q6P5G0', 'MK04_MOUSE'))
        self.assertEqual(record.dr_positive[1023], ('Q16659', 'MK06_HUMAN'))
        self.assertEqual(record.dr_positive[1024], ('Q61532', 'MK06_MOUSE'))
        self.assertEqual(record.dr_positive[1025], ('P27704', 'MK06_RAT'))
        self.assertEqual(record.dr_positive[1026], ('Q13164', 'MK07_HUMAN'))
        self.assertEqual(record.dr_positive[1027], ('Q9WVS8', 'MK07_MOUSE'))
        self.assertEqual(record.dr_positive[1028], ('Q15759', 'MK11_HUMAN'))
        self.assertEqual(record.dr_positive[1029], ('Q9WUI1', 'MK11_MOUSE'))
        self.assertEqual(record.dr_positive[1030], ('O42376', 'MK12_BRARE'))
        self.assertEqual(record.dr_positive[1031], ('P53778', 'MK12_HUMAN'))
        self.assertEqual(record.dr_positive[1032], ('O08911', 'MK12_MOUSE'))
        self.assertEqual(record.dr_positive[1033], ('Q63538', 'MK12_RAT'))
        self.assertEqual(record.dr_positive[1034], ('O15264', 'MK13_HUMAN'))
        self.assertEqual(record.dr_positive[1035], ('Q9Z1B7', 'MK13_MOUSE'))
        self.assertEqual(record.dr_positive[1036], ('Q9N272', 'MK13_PANTR'))
        self.assertEqual(record.dr_positive[1037], ('Q9WTY9', 'MK13_RAT'))
        self.assertEqual(record.dr_positive[1038], ('Q9DGE2', 'MK14A_BRARE'))
        self.assertEqual(record.dr_positive[1039], ('Q90336', 'MK14A_CYPCA'))
        self.assertEqual(record.dr_positive[1040], ('O62618', 'MK14A_DROME'))
        self.assertEqual(record.dr_positive[1041], ('Q9DGE1', 'MK14B_BRARE'))
        self.assertEqual(record.dr_positive[1042], ('Q9I958', 'MK14B_CYPCA'))
        self.assertEqual(record.dr_positive[1043], ('O61443', 'MK14B_DROME'))
        self.assertEqual(record.dr_positive[1044], ('P83100', 'MK14C_DROME'))
        self.assertEqual(record.dr_positive[1045], ('O02812', 'MK14_CANFA'))
        self.assertEqual(record.dr_positive[1046], ('Q16539', 'MK14_HUMAN'))
        self.assertEqual(record.dr_positive[1047], ('P47811', 'MK14_MOUSE'))
        self.assertEqual(record.dr_positive[1048], ('Q95NE7', 'MK14_PANTR'))
        self.assertEqual(record.dr_positive[1049], ('P70618', 'MK14_RAT'))
        self.assertEqual(record.dr_positive[1050], ('P47812', 'MK14_XENLA'))
        self.assertEqual(record.dr_positive[1051], ('Q10407', 'MKH1_SCHPO'))
        self.assertEqual(record.dr_positive[1052], ('P32490', 'MKK1_YEAST'))
        self.assertEqual(record.dr_positive[1053], ('P32491', 'MKK2_YEAST'))
        self.assertEqual(record.dr_positive[1054], ('Q58D94', 'MKNK1_BOVIN'))
        self.assertEqual(record.dr_positive[1055], ('Q9BUB5', 'MKNK1_HUMAN'))
        self.assertEqual(record.dr_positive[1056], ('O08605', 'MKNK1_MOUSE'))
        self.assertEqual(record.dr_positive[1057], ('Q4G050', 'MKNK1_RAT'))
        self.assertEqual(record.dr_positive[1058], ('Q9YGW0', 'MKNK1_XENLA'))
        self.assertEqual(record.dr_positive[1059], ('Q66JF3', 'MKNK1_XENTR'))
        self.assertEqual(record.dr_positive[1060], ('Q9HBH9', 'MKNK2_HUMAN'))
        self.assertEqual(record.dr_positive[1061], ('Q8CDB0', 'MKNK2_MOUSE'))
        self.assertEqual(record.dr_positive[1062], ('Q5U2N4', 'MKNK2_RAT'))
        self.assertEqual(record.dr_positive[1063], ('Q6P431', 'MKNK2_XENLA'))
        self.assertEqual(record.dr_positive[1064], ('Q66I46', 'MKNK2_XENTR'))
        self.assertEqual(record.dr_positive[1065], ('Q07176', 'MMK1_MEDSA'))
        self.assertEqual(record.dr_positive[1066], ('Q40353', 'MMK2_MEDSA'))
        self.assertEqual(record.dr_positive[1067], ('P49657', 'MNB_DROME'))
        self.assertEqual(record.dr_positive[1068], ('Q9UQ07', 'MOK_HUMAN'))
        self.assertEqual(record.dr_positive[1069], ('Q9WVS4', 'MOK_MOUSE'))
        self.assertEqual(record.dr_positive[1070], ('P87347', 'MOS_APTAU'))
        self.assertEqual(record.dr_positive[1071], ('Q8QHF0', 'MOS_ATHNI'))
        self.assertEqual(record.dr_positive[1072], ('Q8AX02', 'MOS_ATHSQ'))
        self.assertEqual(record.dr_positive[1073], ('P10650', 'MOS_CERAE'))
        self.assertEqual(record.dr_positive[1074], ('P10741', 'MOS_CHICK'))
        self.assertEqual(record.dr_positive[1075], ('Q90XV8', 'MOS_CICNG'))
        self.assertEqual(record.dr_positive[1076], ('Q8AX01', 'MOS_DENAN'))
        self.assertEqual(record.dr_positive[1077], ('Q90XV6', 'MOS_GYMCA'))
        self.assertEqual(record.dr_positive[1078], ('P00540', 'MOS_HUMAN'))
        self.assertEqual(record.dr_positive[1079], ('P00536', 'MOS_MOUSE'))
        self.assertEqual(record.dr_positive[1080], ('P07331', 'MOS_MSVMH'))
        self.assertEqual(record.dr_positive[1081], ('P00537', 'MOS_MSVMM'))
        self.assertEqual(record.dr_positive[1082], ('P00538', 'MOS_MSVMO'))
        self.assertEqual(record.dr_positive[1083], ('P32593', 'MOS_MSVMT'))
        self.assertEqual(record.dr_positive[1084], ('P10421', 'MOS_MSVTS'))
        self.assertEqual(record.dr_positive[1085], ('Q90XV9', 'MOS_NYCNY'))
        self.assertEqual(record.dr_positive[1086], ('P50118', 'MOS_PIG'))
        self.assertEqual(record.dr_positive[1087], ('P00539', 'MOS_RAT'))
        self.assertEqual(record.dr_positive[1088], ('Q8AX00', 'MOS_SIBNE'))
        self.assertEqual(record.dr_positive[1089], ('Q90XV7', 'MOS_VULGR'))
        self.assertEqual(record.dr_positive[1090], ('P12965', 'MOS_XENLA'))
        self.assertEqual(record.dr_positive[1091], ('Q63980', 'MP2K1_CRIGR'))
        self.assertEqual(record.dr_positive[1092], ('Q02750', 'MP2K1_HUMAN'))
        self.assertEqual(record.dr_positive[1093], ('P31938', 'MP2K1_MOUSE'))
        self.assertEqual(record.dr_positive[1094], ('Q9XT09', 'MP2K1_PANTR'))
        self.assertEqual(record.dr_positive[1095], ('P29678', 'MP2K1_RABIT'))
        self.assertEqual(record.dr_positive[1096], ('Q01986', 'MP2K1_RAT'))
        self.assertEqual(record.dr_positive[1097], ('Q91447', 'MP2K1_SERCA'))
        self.assertEqual(record.dr_positive[1098], ('Q05116', 'MP2K1_XENLA'))
        self.assertEqual(record.dr_positive[1099], ('Q90891', 'MP2K2_CHICK'))
        self.assertEqual(record.dr_positive[1100], ('Q90321', 'MP2K2_CYPCA'))
        self.assertEqual(record.dr_positive[1101], ('P36507', 'MP2K2_HUMAN'))
        self.assertEqual(record.dr_positive[1102], ('Q63932', 'MP2K2_MOUSE'))
        self.assertEqual(record.dr_positive[1103], ('P36506', 'MP2K2_RAT'))
        self.assertEqual(record.dr_positive[1104], ('Q07192', 'MP2K2_XENLA'))
        self.assertEqual(record.dr_positive[1105], ('P46734', 'MP2K3_HUMAN'))
        self.assertEqual(record.dr_positive[1106], ('O09110', 'MP2K3_MOUSE'))
        self.assertEqual(record.dr_positive[1107], ('P45985', 'MP2K4_HUMAN'))
        self.assertEqual(record.dr_positive[1108], ('P47809', 'MP2K4_MOUSE'))
        self.assertEqual(record.dr_positive[1109], ('Q13163', 'MP2K5_HUMAN'))
        self.assertEqual(record.dr_positive[1110], ('Q9WVS7', 'MP2K5_MOUSE'))
        self.assertEqual(record.dr_positive[1111], ('Q62862', 'MP2K5_RAT'))
        self.assertEqual(record.dr_positive[1112], ('P52564', 'MP2K6_HUMAN'))
        self.assertEqual(record.dr_positive[1113], ('P70236', 'MP2K6_MOUSE'))
        self.assertEqual(record.dr_positive[1114], ('O94235', 'MPH1_SCHPO'))
        self.assertEqual(record.dr_positive[1115], ('Q39021', 'MPK1_ARATH'))
        self.assertEqual(record.dr_positive[1116], ('Q39022', 'MPK2_ARATH'))
        self.assertEqual(record.dr_positive[1117], ('Q39023', 'MPK3_ARATH'))
        self.assertEqual(record.dr_positive[1118], ('Q39024', 'MPK4_ARATH'))
        self.assertEqual(record.dr_positive[1119], ('Q39025', 'MPK5_ARATH'))
        self.assertEqual(record.dr_positive[1120], ('Q39026', 'MPK6_ARATH'))
        self.assertEqual(record.dr_positive[1121], ('Q39027', 'MPK7_ARATH'))
        self.assertEqual(record.dr_positive[1122], ('Q8AYG3', 'MPS1_BRARE'))
        self.assertEqual(record.dr_positive[1123], ('P54199', 'MPS1_YEAST'))
        self.assertEqual(record.dr_positive[1124], ('Q5VT25', 'MRCKA_HUMAN'))
        self.assertEqual(record.dr_positive[1125], ('O54874', 'MRCKA_RAT'))
        self.assertEqual(record.dr_positive[1126], ('Q9Y5S2', 'MRCKB_HUMAN'))
        self.assertEqual(record.dr_positive[1127], ('Q7TT50', 'MRCKB_MOUSE'))
        self.assertEqual(record.dr_positive[1128], ('Q7TT49', 'MRCKB_RAT'))
        self.assertEqual(record.dr_positive[1129], ('Q6DT37', 'MRCKG_HUMAN'))
        self.assertEqual(record.dr_positive[1130], ('Q80UW5', 'MRCKG_MOUSE'))
        self.assertEqual(record.dr_positive[1131], ('P50873', 'MRK1_YEAST'))
        self.assertEqual(record.dr_positive[1132], ('Q6UR05', 'MRP1_CANFA'))
        self.assertEqual(record.dr_positive[1133], ('Q5F364', 'MRP1_CHICK'))
        self.assertEqual(record.dr_positive[1134], ('Q864R9', 'MRP1_MACFA'))
        self.assertEqual(record.dr_positive[1135], ('P51137', 'MSK1_MEDSA'))
        self.assertEqual(record.dr_positive[1136], ('Q40518', 'MSK1_TOBAC'))
        self.assertEqual(record.dr_positive[1137], ('P51138', 'MSK2_MEDSA'))
        self.assertEqual(record.dr_positive[1138], ('P51139', 'MSK3_MEDSA'))
        self.assertEqual(record.dr_positive[1139], ('Q9P289', 'MST4_HUMAN'))
        self.assertEqual(record.dr_positive[1140], ('Q99JT2', 'MST4_MOUSE'))
        self.assertEqual(record.dr_positive[1141], ('Q8AXY6', 'MUSK_CHICK'))
        self.assertEqual(record.dr_positive[1142], ('O15146', 'MUSK_HUMAN'))
        self.assertEqual(record.dr_positive[1143], ('Q61006', 'MUSK_MOUSE'))
        self.assertEqual(record.dr_positive[1144], ('Q62838', 'MUSK_RAT'))
        self.assertEqual(record.dr_positive[1145], ('Q9H1R3', 'MYLK2_HUMAN'))
        self.assertEqual(record.dr_positive[1146], ('P07313', 'MYLK2_RABIT'))
        self.assertEqual(record.dr_positive[1147], ('P20689', 'MYLK2_RAT'))
        self.assertEqual(record.dr_positive[1148], ('Q28824', 'MYLK_BOVIN'))
        self.assertEqual(record.dr_positive[1149], ('P11799', 'MYLK_CHICK'))
        self.assertEqual(record.dr_positive[1150], ('P25323', 'MYLK_DICDI'))
        self.assertEqual(record.dr_positive[1151], ('Q15746', 'MYLK_HUMAN'))
        self.assertEqual(record.dr_positive[1152], ('P29294', 'MYLK_RABIT'))
        self.assertEqual(record.dr_positive[1153], ('Q8NEV4', 'MYO3A_HUMAN'))
        self.assertEqual(record.dr_positive[1154], ('Q8K3H5', 'MYO3A_MOUSE'))
        self.assertEqual(record.dr_positive[1155], ('Q8WXR4', 'MYO3B_HUMAN'))
        self.assertEqual(record.dr_positive[1156], ('O75011', 'NAK1_SCHPO'))
        self.assertEqual(record.dr_positive[1157], ('P43293', 'NAK_ARATH'))
        self.assertEqual(record.dr_positive[1158], ('P84199', 'NEK1_CAEEL'))
        self.assertEqual(record.dr_positive[1159], ('Q96PY6', 'NEK1_HUMAN'))
        self.assertEqual(record.dr_positive[1160], ('P51954', 'NEK1_MOUSE'))
        self.assertEqual(record.dr_positive[1161], ('P51956', 'NEK3_HUMAN'))
        self.assertEqual(record.dr_positive[1162], ('Q9R0A5', 'NEK3_MOUSE'))
        self.assertEqual(record.dr_positive[1163], ('P51957', 'NEK4_HUMAN'))
        self.assertEqual(record.dr_positive[1164], ('Q9Z1J2', 'NEK4_MOUSE'))
        self.assertEqual(record.dr_positive[1165], ('Q9HC98', 'NEK6_HUMAN'))
        self.assertEqual(record.dr_positive[1166], ('Q9ES70', 'NEK6_MOUSE'))
        self.assertEqual(record.dr_positive[1167], ('P59895', 'NEK6_RAT'))
        self.assertEqual(record.dr_positive[1168], ('Q8TDX7', 'NEK7_HUMAN'))
        self.assertEqual(record.dr_positive[1169], ('Q9ES74', 'NEK7_MOUSE'))
        self.assertEqual(record.dr_positive[1170], ('Q90XC2', 'NEK8_BRARE'))
        self.assertEqual(record.dr_positive[1171], ('Q86SG6', 'NEK8_HUMAN'))
        self.assertEqual(record.dr_positive[1172], ('Q91ZR4', 'NEK8_MOUSE'))
        self.assertEqual(record.dr_positive[1173], ('Q7KRY6', 'NHK1_DROME'))
        self.assertEqual(record.dr_positive[1174], ('Q9UBE8', 'NLK_HUMAN'))
        self.assertEqual(record.dr_positive[1175], ('O54949', 'NLK_MOUSE'))
        self.assertEqual(record.dr_positive[1176], ('Q8L4H4', 'NORK_MEDTR'))
        self.assertEqual(record.dr_positive[1177], ('Q8LKZ1', 'NORK_PEA'))
        self.assertEqual(record.dr_positive[1178], ('P22211', 'NPR1_YEAST'))
        self.assertEqual(record.dr_positive[1179], ('Q08942', 'NRKA_TRYBB'))
        self.assertEqual(record.dr_positive[1180], ('Q40517', 'NTF3_TOBAC'))
        self.assertEqual(record.dr_positive[1181], ('Q40532', 'NTF4_TOBAC'))
        self.assertEqual(record.dr_positive[1182], ('Q40531', 'NTF6_TOBAC'))
        self.assertEqual(record.dr_positive[1183], ('Q91009', 'NTRK1_CHICK'))
        self.assertEqual(record.dr_positive[1184], ('P04629', 'NTRK1_HUMAN'))
        self.assertEqual(record.dr_positive[1185], ('P35739', 'NTRK1_RAT'))
        self.assertEqual(record.dr_positive[1186], ('Q91987', 'NTRK2_CHICK'))
        self.assertEqual(record.dr_positive[1187], ('Q16620', 'NTRK2_HUMAN'))
        self.assertEqual(record.dr_positive[1188], ('P15209', 'NTRK2_MOUSE'))
        self.assertEqual(record.dr_positive[1189], ('Q63604', 'NTRK2_RAT'))
        self.assertEqual(record.dr_positive[1190], ('Q91044', 'NTRK3_CHICK'))
        self.assertEqual(record.dr_positive[1191], ('Q16288', 'NTRK3_HUMAN'))
        self.assertEqual(record.dr_positive[1192], ('Q5IFJ9', 'NTRK3_MACFA'))
        self.assertEqual(record.dr_positive[1193], ('Q5IS37', 'NTRK3_PANTR'))
        self.assertEqual(record.dr_positive[1194], ('P24786', 'NTRK3_PIG'))
        self.assertEqual(record.dr_positive[1195], ('Q03351', 'NTRK3_RAT'))
        self.assertEqual(record.dr_positive[1196], ('Q5IS82', 'NTRK3_SAIBB'))
        self.assertEqual(record.dr_positive[1197], ('O60285', 'NUAK1_HUMAN'))
        self.assertEqual(record.dr_positive[1198], ('Q641K5', 'NUAK1_MOUSE'))
        self.assertEqual(record.dr_positive[1199], ('O13310', 'ORB6_SCHPO'))
        self.assertEqual(record.dr_positive[1200], ('O95747', 'OXSR1_HUMAN'))
        self.assertEqual(record.dr_positive[1201], ('Q6P9R2', 'OXSR1_MOUSE'))
        self.assertEqual(record.dr_positive[1202], ('Q863I2', 'OXSR1_PIG'))
        self.assertEqual(record.dr_positive[1203], ('Q5R495', 'OXSR1_PONPY'))
        self.assertEqual(record.dr_positive[1204], ('Q17850', 'PAK1_CAEEL'))
        self.assertEqual(record.dr_positive[1205], ('Q13153', 'PAK1_HUMAN'))
        self.assertEqual(record.dr_positive[1206], ('O88643', 'PAK1_MOUSE'))
        self.assertEqual(record.dr_positive[1207], ('P35465', 'PAK1_RAT'))
        self.assertEqual(record.dr_positive[1208], ('P38990', 'PAK1_YEAST'))
        self.assertEqual(record.dr_positive[1209], ('Q13177', 'PAK2_HUMAN'))
        self.assertEqual(record.dr_positive[1210], ('Q8CIN4', 'PAK2_MOUSE'))
        self.assertEqual(record.dr_positive[1211], ('Q29502', 'PAK2_RABIT'))
        self.assertEqual(record.dr_positive[1212], ('Q64303', 'PAK2_RAT'))
        self.assertEqual(record.dr_positive[1213], ('O75914', 'PAK3_HUMAN'))
        self.assertEqual(record.dr_positive[1214], ('Q61036', 'PAK3_MOUSE'))
        self.assertEqual(record.dr_positive[1215], ('Q7YQL4', 'PAK3_PANTR'))
        self.assertEqual(record.dr_positive[1216], ('Q7YQL3', 'PAK3_PONPY'))
        self.assertEqual(record.dr_positive[1217], ('Q62829', 'PAK3_RAT'))
        self.assertEqual(record.dr_positive[1218], ('O96013', 'PAK4_HUMAN'))
        self.assertEqual(record.dr_positive[1219], ('Q8BTW9', 'PAK4_MOUSE'))
        self.assertEqual(record.dr_positive[1220], ('Q9NQU5', 'PAK6_HUMAN'))
        self.assertEqual(record.dr_positive[1221], ('Q9P286', 'PAK7_HUMAN'))
        self.assertEqual(record.dr_positive[1222], ('Q8C015', 'PAK7_MOUSE'))
        self.assertEqual(record.dr_positive[1223], ('Q9VXE5', 'PAKM_DROME'))
        self.assertEqual(record.dr_positive[1224], ('Q96RG2', 'PASK_HUMAN'))
        self.assertEqual(record.dr_positive[1225], ('Q8CEE6', 'PASK_MOUSE'))
        self.assertEqual(record.dr_positive[1226], ('Q9FE20', 'PBS1_ARATH'))
        self.assertEqual(record.dr_positive[1227], ('P36582', 'PCK1_SCHPO'))
        self.assertEqual(record.dr_positive[1228], ('P36583', 'PCK2_SCHPO'))
        self.assertEqual(record.dr_positive[1229], ('Q00536', 'PCTK1_HUMAN'))
        self.assertEqual(record.dr_positive[1230], ('Q04735', 'PCTK1_MOUSE'))
        self.assertEqual(record.dr_positive[1231], ('Q63686', 'PCTK1_RAT'))
        self.assertEqual(record.dr_positive[1232], ('Q00537', 'PCTK2_HUMAN'))
        self.assertEqual(record.dr_positive[1233], ('Q8K0D0', 'PCTK2_MOUSE'))
        self.assertEqual(record.dr_positive[1234], ('O35831', 'PCTK2_RAT'))
        self.assertEqual(record.dr_positive[1235], ('Q07002', 'PCTK3_HUMAN'))
        self.assertEqual(record.dr_positive[1236], ('Q04899', 'PCTK3_MOUSE'))
        self.assertEqual(record.dr_positive[1237], ('Q5RD01', 'PCTK3_PONPY'))
        self.assertEqual(record.dr_positive[1238], ('O35832', 'PCTK3_RAT'))
        self.assertEqual(record.dr_positive[1239], ('Q6NU47', 'PDK1A_XENLA'))
        self.assertEqual(record.dr_positive[1240], ('Q6NU98', 'PDK1B_XENLA'))
        self.assertEqual(record.dr_positive[1241], ('Q8N165', 'PDK1L_HUMAN'))
        self.assertEqual(record.dr_positive[1242], ('Q8QZR7', 'PDK1L_MOUSE'))
        self.assertEqual(record.dr_positive[1243], ('Q5RER6', 'PDK1L_PONPY'))
        self.assertEqual(record.dr_positive[1244], ('Q9W0V1', 'PDPK1_DROME'))
        self.assertEqual(record.dr_positive[1245], ('O15530', 'PDPK1_HUMAN'))
        self.assertEqual(record.dr_positive[1246], ('Q9Z2A0', 'PDPK1_MOUSE'))
        self.assertEqual(record.dr_positive[1247], ('O55173', 'PDPK1_RAT'))
        self.assertEqual(record.dr_positive[1248], ('O74456', 'PEF1_SCHPO'))
        self.assertEqual(record.dr_positive[1249], ('O94921', 'PFTK1_HUMAN'))
        self.assertEqual(record.dr_positive[1250], ('O35495', 'PFTK1_MOUSE'))
        self.assertEqual(record.dr_positive[1251], ('P16234', 'PGFRA_HUMAN'))
        self.assertEqual(record.dr_positive[1252], ('P26618', 'PGFRA_MOUSE'))
        self.assertEqual(record.dr_positive[1253], ('P20786', 'PGFRA_RAT'))
        self.assertEqual(record.dr_positive[1254], ('P26619', 'PGFRA_XENLA'))
        self.assertEqual(record.dr_positive[1255], ('Q6QNF3', 'PGFRB_CANFA'))
        self.assertEqual(record.dr_positive[1256], ('P09619', 'PGFRB_HUMAN'))
        self.assertEqual(record.dr_positive[1257], ('P05622', 'PGFRB_MOUSE'))
        self.assertEqual(record.dr_positive[1258], ('Q05030', 'PGFRB_RAT'))
        self.assertEqual(record.dr_positive[1259], ('Q16816', 'PHKG1_HUMAN'))
        self.assertEqual(record.dr_positive[1260], ('P07934', 'PHKG1_MOUSE'))
        self.assertEqual(record.dr_positive[1261], ('P00518', 'PHKG1_RABIT'))
        self.assertEqual(record.dr_positive[1262], ('P13286', 'PHKG1_RAT'))
        self.assertEqual(record.dr_positive[1263], ('P15735', 'PHKG2_HUMAN'))
        self.assertEqual(record.dr_positive[1264], ('Q9DB30', 'PHKG2_MOUSE'))
        self.assertEqual(record.dr_positive[1265], ('P31325', 'PHKG2_RAT'))
        self.assertEqual(record.dr_positive[1266], ('Q751E8', 'PHO85_ASHGO'))
        self.assertEqual(record.dr_positive[1267], ('Q9HGY5', 'PHO85_CANAL'))
        self.assertEqual(record.dr_positive[1268], ('Q6FKD4', 'PHO85_CANGA'))
        self.assertEqual(record.dr_positive[1269], ('Q6BRY2', 'PHO85_DEBHA'))
        self.assertEqual(record.dr_positive[1270], ('Q92241', 'PHO85_KLULA'))
        self.assertEqual(record.dr_positive[1271], ('Q6C7U8', 'PHO85_YARLI'))
        self.assertEqual(record.dr_positive[1272], ('P17157', 'PHO85_YEAST'))
        self.assertEqual(record.dr_positive[1273], ('O48963', 'PHOT1_ARATH'))
        self.assertEqual(record.dr_positive[1274], ('P93025', 'PHOT2_ARATH'))
        self.assertEqual(record.dr_positive[1275], ('P25848', 'PHY1_CERPU'))
        self.assertEqual(record.dr_positive[1276], ('Q9N0P9', 'PIM1_BOVIN'))
        self.assertEqual(record.dr_positive[1277], ('Q9YHZ5', 'PIM1_BRARE'))
        self.assertEqual(record.dr_positive[1278], ('Q95LJ0', 'PIM1_FELCA'))
        self.assertEqual(record.dr_positive[1279], ('P11309', 'PIM1_HUMAN'))
        self.assertEqual(record.dr_positive[1280], ('P06803', 'PIM1_MOUSE'))
        self.assertEqual(record.dr_positive[1281], ('P26794', 'PIM1_RAT'))
        self.assertEqual(record.dr_positive[1282], ('Q9P1W9', 'PIM2_HUMAN'))
        self.assertEqual(record.dr_positive[1283], ('Q62070', 'PIM2_MOUSE'))
        self.assertEqual(record.dr_positive[1284], ('Q9PU85', 'PIM3_COTJA'))
        self.assertEqual(record.dr_positive[1285], ('Q86V86', 'PIM3_HUMAN'))
        self.assertEqual(record.dr_positive[1286], ('P58750', 'PIM3_MOUSE'))
        self.assertEqual(record.dr_positive[1287], ('O70444', 'PIM3_RAT'))
        self.assertEqual(record.dr_positive[1288], ('Q99MQ3', 'PINK1_MOUSE'))
        self.assertEqual(record.dr_positive[1289], ('O14132', 'PIT1_SCHPO'))
        self.assertEqual(record.dr_positive[1290], ('P28178', 'PK2_DICDI'))
        self.assertEqual(record.dr_positive[1291], ('Q03407', 'PKH1_YEAST'))
        self.assertEqual(record.dr_positive[1292], ('Q753D9', 'PKH3_ASHGO'))
        self.assertEqual(record.dr_positive[1293], ('Q03306', 'PKH3_YEAST'))
        self.assertEqual(record.dr_positive[1294], ('P37562', 'PKN1_BACSU'))
        self.assertEqual(record.dr_positive[1295], ('Q8FUI5', 'PKN1_COREF'))
        self.assertEqual(record.dr_positive[1296], ('Q8NU98', 'PKN1_CORGL'))
        self.assertEqual(record.dr_positive[1297], ('Q16512', 'PKN1_HUMAN'))
        self.assertEqual(record.dr_positive[1298], ('P70268', 'PKN1_MOUSE'))
        self.assertEqual(record.dr_positive[1299], ('P33973', 'PKN1_MYXXA'))
        self.assertEqual(record.dr_positive[1300], ('Q63433', 'PKN1_RAT'))
        self.assertEqual(record.dr_positive[1301], ('Q8R9T6', 'PKN1_THETN'))
        self.assertEqual(record.dr_positive[1302], ('O34507', 'PKN2_BACSU'))
        self.assertEqual(record.dr_positive[1303], ('Q97IC2', 'PKN2_CLOAB'))
        self.assertEqual(record.dr_positive[1304], ('Q8XJL8', 'PKN2_CLOPE'))
        self.assertEqual(record.dr_positive[1305], ('Q8FUI4', 'PKN2_COREF'))
        self.assertEqual(record.dr_positive[1306], ('Q8NU97', 'PKN2_CORGL'))
        self.assertEqual(record.dr_positive[1307], ('Q16513', 'PKN2_HUMAN'))
        self.assertEqual(record.dr_positive[1308], ('Q8BWW9', 'PKN2_MOUSE'))
        self.assertEqual(record.dr_positive[1309], ('P54736', 'PKN2_MYXXA'))
        self.assertEqual(record.dr_positive[1310], ('O08874', 'PKN2_RAT'))
        self.assertEqual(record.dr_positive[1311], ('Q6P5Z2', 'PKN3_HUMAN'))
        self.assertEqual(record.dr_positive[1312], ('Q8K045', 'PKN3_MOUSE'))
        self.assertEqual(record.dr_positive[1313], ('Q9XBQ0', 'PKN3_MYXXA'))
        self.assertEqual(record.dr_positive[1314], ('P54734', 'PKNA_ANASP'))
        self.assertEqual(record.dr_positive[1315], ('Q8G6P9', 'PKNB_BIFLO'))
        self.assertEqual(record.dr_positive[1316], ('Q9CEF5', 'PKNB_LACLA'))
        self.assertEqual(record.dr_positive[1317], ('P0A5S5', 'PKNB_MYCBO'))
        self.assertEqual(record.dr_positive[1318], ('P54744', 'PKNB_MYCLE'))
        self.assertEqual(record.dr_positive[1319], ('P0A5S4', 'PKNB_MYCTU'))
        self.assertEqual(record.dr_positive[1320], ('Q822K5', 'PKND_CHLCV'))
        self.assertEqual(record.dr_positive[1321], ('Q9PK92', 'PKND_CHLMU'))
        self.assertEqual(record.dr_positive[1322], ('Q9Z986', 'PKND_CHLPN'))
        self.assertEqual(record.dr_positive[1323], ('O84303', 'PKND_CHLTR'))
        self.assertEqual(record.dr_positive[1324], ('O05871', 'PKND_MYCTU'))
        self.assertEqual(record.dr_positive[1325], ('Q7TZN3', 'PKNE_MYCBO'))
        self.assertEqual(record.dr_positive[1326], ('P72001', 'PKNE_MYCTU'))
        self.assertEqual(record.dr_positive[1327], ('Q7U095', 'PKNH_MYCBO'))
        self.assertEqual(record.dr_positive[1328], ('Q11053', 'PKNH_MYCTU'))
        self.assertEqual(record.dr_positive[1329], ('Q7TXA9', 'PKNK_MYCBO'))
        self.assertEqual(record.dr_positive[1330], ('P95078', 'PKNK_MYCTU'))
        self.assertEqual(record.dr_positive[1331], ('Q9XA16', 'PKNX_STRCO'))
        self.assertEqual(record.dr_positive[1332], ('Q9S2C0', 'PKSC_STRCO'))
        self.assertEqual(record.dr_positive[1333], ('P49695', 'PKWA_THECU'))
        self.assertEqual(record.dr_positive[1334], ('P34331', 'PLK1_CAEEL'))
        self.assertEqual(record.dr_positive[1335], ('P53350', 'PLK1_HUMAN'))
        self.assertEqual(record.dr_positive[1336], ('Q07832', 'PLK1_MOUSE'))
        self.assertEqual(record.dr_positive[1337], ('Q62673', 'PLK1_RAT'))
        self.assertEqual(record.dr_positive[1338], ('P70032', 'PLK1_XENLA'))
        self.assertEqual(record.dr_positive[1339], ('P62205', 'PLK1_XENTR'))
        self.assertEqual(record.dr_positive[1340], ('Q9N2L7', 'PLK2_CAEEL'))
        self.assertEqual(record.dr_positive[1341], ('Q9NYY3', 'PLK2_HUMAN'))
        self.assertEqual(record.dr_positive[1342], ('P53351', 'PLK2_MOUSE'))
        self.assertEqual(record.dr_positive[1343], ('Q9R012', 'PLK2_RAT'))
        self.assertEqual(record.dr_positive[1344], ('Q20845', 'PLK3_CAEEL'))
        self.assertEqual(record.dr_positive[1345], ('Q9H4B4', 'PLK3_HUMAN'))
        self.assertEqual(record.dr_positive[1346], ('Q60806', 'PLK3_MOUSE'))
        self.assertEqual(record.dr_positive[1347], ('Q9R011', 'PLK3_RAT'))
        self.assertEqual(record.dr_positive[1348], ('O00444', 'PLK4_HUMAN'))
        self.assertEqual(record.dr_positive[1349], ('Q64702', 'PLK4_MOUSE'))
        self.assertEqual(record.dr_positive[1350], ('P50528', 'PLO1_SCHPO'))
        self.assertEqual(record.dr_positive[1351], ('Q17446', 'PMK1_CAEEL'))
        self.assertEqual(record.dr_positive[1352], ('Q8MXI4', 'PMK2_CAEEL'))
        self.assertEqual(record.dr_positive[1353], ('O44514', 'PMK3_CAEEL'))
        self.assertEqual(record.dr_positive[1354], ('O18209', 'PMYT1_CAEEL'))
        self.assertEqual(record.dr_positive[1355], ('Q9NI63', 'PMYT1_DROME'))
        self.assertEqual(record.dr_positive[1356], ('Q99640', 'PMYT1_HUMAN'))
        self.assertEqual(record.dr_positive[1357], ('Q9ESG9', 'PMYT1_MOUSE'))
        self.assertEqual(record.dr_positive[1358], ('Q91618', 'PMYT1_XENLA'))
        self.assertEqual(record.dr_positive[1359], ('Q09690', 'POM1_SCHPO'))
        self.assertEqual(record.dr_positive[1360], ('Q9I7F7', 'PR2_DROME'))
        self.assertEqual(record.dr_positive[1361], ('P51817', 'PRKX_HUMAN'))
        self.assertEqual(record.dr_positive[1362], ('Q922R0', 'PRKX_MOUSE'))
        self.assertEqual(record.dr_positive[1363], ('Q07538', 'PRP4_SCHPO'))
        self.assertEqual(record.dr_positive[1364], ('Q9ZVR7', 'PSKR_ARATH'))
        self.assertEqual(record.dr_positive[1365], ('Q8LPB4', 'PSKR_DAUCA'))
        self.assertEqual(record.dr_positive[1366], ('Q6FRE7', 'PTK2_CANGA'))
        self.assertEqual(record.dr_positive[1367], ('P47116', 'PTK2_YEAST'))
        self.assertEqual(record.dr_positive[1368], ('Q13882', 'PTK6_HUMAN'))
        self.assertEqual(record.dr_positive[1369], ('Q64434', 'PTK6_MOUSE'))
        self.assertEqual(record.dr_positive[1370], ('Q9CZE3', 'RAB32_MOUSE'))
        self.assertEqual(record.dr_positive[1371], ('P22216', 'RAD53_YEAST'))
        self.assertEqual(record.dr_positive[1372], ('P05625', 'RAF1_CHICK'))
        self.assertEqual(record.dr_positive[1373], ('P04049', 'RAF1_HUMAN'))
        self.assertEqual(record.dr_positive[1374], ('Q99N57', 'RAF1_MOUSE'))
        self.assertEqual(record.dr_positive[1375], ('P11345', 'RAF1_RAT'))
        self.assertEqual(record.dr_positive[1376], ('P09560', 'RAF1_XENLA'))
        self.assertEqual(record.dr_positive[1377], ('P00532', 'RAF_MSV36'))
        self.assertEqual(record.dr_positive[1378], ('P40230', 'RAG8_KLULA'))
        self.assertEqual(record.dr_positive[1379], ('P08092', 'RAN1_SCHPO'))
        self.assertEqual(record.dr_positive[1380], ('P38622', 'RCK1_YEAST'))
        self.assertEqual(record.dr_positive[1381], ('P38623', 'RCK2_YEAST'))
        self.assertEqual(record.dr_positive[1382], ('P07949', 'RET_HUMAN'))
        self.assertEqual(record.dr_positive[1383], ('P35546', 'RET_MOUSE'))
        self.assertEqual(record.dr_positive[1384], ('Q9QZL0', 'RIPK3_MOUSE'))
        self.assertEqual(record.dr_positive[1385], ('Q9Z2P5', 'RIPK3_RAT'))
        self.assertEqual(record.dr_positive[1386], ('P57078', 'RIPK4_HUMAN'))
        self.assertEqual(record.dr_positive[1387], ('Q02723', 'RKIN1_SECCE'))
        self.assertEqual(record.dr_positive[1388], ('P28327', 'RK_BOVIN'))
        self.assertEqual(record.dr_positive[1389], ('Q15835', 'RK_HUMAN'))
        self.assertEqual(record.dr_positive[1390], ('Q9WVL4', 'RK_MOUSE'))
        self.assertEqual(record.dr_positive[1391], ('Q63651', 'RK_RAT'))
        self.assertEqual(record.dr_positive[1392], ('Q9LQQ8', 'RLCK7_ARATH'))
        self.assertEqual(record.dr_positive[1393], ('P47735', 'RLK5_ARATH'))
        self.assertEqual(record.dr_positive[1394], ('P27966', 'RMIL_AVEVR'))
        self.assertEqual(record.dr_positive[1395], ('P10533', 'RMIL_AVII1'))
        self.assertEqual(record.dr_positive[1396], ('Q13464', 'ROCK1_HUMAN'))
        self.assertEqual(record.dr_positive[1397], ('P70335', 'ROCK1_MOUSE'))
        self.assertEqual(record.dr_positive[1398], ('O77819', 'ROCK1_RABIT'))
        self.assertEqual(record.dr_positive[1399], ('Q63644', 'ROCK1_RAT'))
        self.assertEqual(record.dr_positive[1400], ('Q28021', 'ROCK2_BOVIN'))
        self.assertEqual(record.dr_positive[1401], ('O75116', 'ROCK2_HUMAN'))
        self.assertEqual(record.dr_positive[1402], ('P70336', 'ROCK2_MOUSE'))
        self.assertEqual(record.dr_positive[1403], ('Q62868', 'ROCK2_RAT'))
        self.assertEqual(record.dr_positive[1404], ('Q04912', 'RON_HUMAN'))
        self.assertEqual(record.dr_positive[1405], ('Q62190', 'RON_MOUSE'))
        self.assertEqual(record.dr_positive[1406], ('Q24488', 'ROR1_DROME'))
        self.assertEqual(record.dr_positive[1407], ('Q9V6K3', 'ROR2_DROME'))
        self.assertEqual(record.dr_positive[1408], ('P00529', 'ROS_AVISU'))
        self.assertEqual(record.dr_positive[1409], ('P08941', 'ROS_CHICK'))
        self.assertEqual(record.dr_positive[1410], ('P08922', 'ROS_HUMAN'))
        self.assertEqual(record.dr_positive[1411], ('P93194', 'RPK1_IPONI'))
        self.assertEqual(record.dr_positive[1412], ('P33497', 'RYK_AVIR3'))
        self.assertEqual(record.dr_positive[1413], ('Q75LR7', 'SAPK1_ORYSA'))
        self.assertEqual(record.dr_positive[1414], ('Q84TC6', 'SAPK2_ORYSA'))
        self.assertEqual(record.dr_positive[1415], ('Q75V63', 'SAPK3_ORYSA'))
        self.assertEqual(record.dr_positive[1416], ('Q5N942', 'SAPK4_ORYSA'))
        self.assertEqual(record.dr_positive[1417], ('Q7XKA8', 'SAPK5_ORYSA'))
        self.assertEqual(record.dr_positive[1418], ('Q6ZI44', 'SAPK6_ORYSA'))
        self.assertEqual(record.dr_positive[1419], ('Q7XQP4', 'SAPK7_ORYSA'))
        self.assertEqual(record.dr_positive[1420], ('Q7Y0B9', 'SAPK8_ORYSA'))
        self.assertEqual(record.dr_positive[1421], ('Q75V57', 'SAPK9_ORYSA'))
        self.assertEqual(record.dr_positive[1422], ('Q75H77', 'SAPKA_ORYSA'))
        self.assertEqual(record.dr_positive[1423], ('P11792', 'SCH9_YEAST'))
        self.assertEqual(record.dr_positive[1424], ('P50530', 'SCK1_SCHPO'))
        self.assertEqual(record.dr_positive[1425], ('Q10364', 'SCK2_SCHPO'))
        self.assertEqual(record.dr_positive[1426], ('P23049', 'SEA_AVIET'))
        self.assertEqual(record.dr_positive[1427], ('P18431', 'SGG_DROME'))
        self.assertEqual(record.dr_positive[1428], ('O00141', 'SGK1_HUMAN'))
        self.assertEqual(record.dr_positive[1429], ('Q9WVC6', 'SGK1_MOUSE'))
        self.assertEqual(record.dr_positive[1430], ('Q9XT18', 'SGK1_RABIT'))
        self.assertEqual(record.dr_positive[1431], ('Q06226', 'SGK1_RAT'))
        self.assertEqual(record.dr_positive[1432], ('Q9HBY8', 'SGK2_HUMAN'))
        self.assertEqual(record.dr_positive[1433], ('Q9QZS5', 'SGK2_MOUSE'))
        self.assertEqual(record.dr_positive[1434], ('Q8R4U9', 'SGK2_RAT'))
        self.assertEqual(record.dr_positive[1435], ('Q96BR1', 'SGK3_HUMAN'))
        self.assertEqual(record.dr_positive[1436], ('Q9ERE3', 'SGK3_MOUSE'))
        self.assertEqual(record.dr_positive[1437], ('Q24145', 'SHARK_DROME'))
        self.assertEqual(record.dr_positive[1438], ('P50527', 'SHK1_SCHPO'))
        self.assertEqual(record.dr_positive[1439], ('Q10056', 'SHK2_SCHPO'))
        self.assertEqual(record.dr_positive[1440], ('O14305', 'SID1_SCHPO'))
        self.assertEqual(record.dr_positive[1441], ('Q09898', 'SID2_SCHPO'))
        self.assertEqual(record.dr_positive[1442], ('O64483', 'SIRK_ARATH'))
        self.assertEqual(record.dr_positive[1443], ('Q9Y884', 'SKH1_SCHPO'))
        self.assertEqual(record.dr_positive[1444], ('Q12505', 'SKS1_YEAST'))
        self.assertEqual(record.dr_positive[1445], ('Q03656', 'SKY1_YEAST'))
        self.assertEqual(record.dr_positive[1446], ('Q00772', 'SLT2_YEAST'))
        self.assertEqual(record.dr_positive[1447], ('Q09488', 'SMA6_CAEEL'))
        self.assertEqual(record.dr_positive[1448], ('P41808', 'SMK1_YEAST'))
        self.assertEqual(record.dr_positive[1449], ('P57059', 'SN1L1_HUMAN'))
        self.assertEqual(record.dr_positive[1450], ('Q60670', 'SN1L1_MOUSE'))
        self.assertEqual(record.dr_positive[1451], ('Q9R1U5', 'SN1L1_RAT'))
        self.assertEqual(record.dr_positive[1452], ('Q9IA88', 'SN1L2_CHICK'))
        self.assertEqual(record.dr_positive[1453], ('Q9H0K1', 'SN1L2_HUMAN'))
        self.assertEqual(record.dr_positive[1454], ('Q8CFH6', 'SN1L2_MOUSE'))
        self.assertEqual(record.dr_positive[1455], ('Q5REX1', 'SN1L2_PONPY'))
        self.assertEqual(record.dr_positive[1456], ('P52497', 'SNF1_CANAL'))
        self.assertEqual(record.dr_positive[1457], ('Q00372', 'SNF1_CANGA'))
        self.assertEqual(record.dr_positive[1458], ('O94168', 'SNF1_CANTR'))
        self.assertEqual(record.dr_positive[1459], ('O74536', 'SNF1_SCHPO'))
        self.assertEqual(record.dr_positive[1460], ('P06782', 'SNF1_YEAST'))
        self.assertEqual(record.dr_positive[1461], ('Q9NRH2', 'SNRK_HUMAN'))
        self.assertEqual(record.dr_positive[1462], ('Q8VDU5', 'SNRK_MOUSE'))
        self.assertEqual(record.dr_positive[1463], ('Q63553', 'SNRK_RAT'))
        self.assertEqual(record.dr_positive[1464], ('Q61IS6', 'SPK1_CAEBR'))
        self.assertEqual(record.dr_positive[1465], ('Q03563', 'SPK1_CAEEL'))
        self.assertEqual(record.dr_positive[1466], ('P42687', 'SPK1_DUGTI'))
        self.assertEqual(record.dr_positive[1467], ('P27638', 'SPK1_SCHPO'))
        self.assertEqual(record.dr_positive[1468], ('Q9FAB3', 'SPKA_SYNY3'))
        self.assertEqual(record.dr_positive[1469], ('P74297', 'SPKB_SYNY3'))
        self.assertEqual(record.dr_positive[1470], ('P74745', 'SPKC_SYNY3'))
        self.assertEqual(record.dr_positive[1471], ('P54735', 'SPKD_SYNY3'))
        self.assertEqual(record.dr_positive[1472], ('P73469', 'SPKF_SYNY3'))
        self.assertEqual(record.dr_positive[1473], ('Q92398', 'SPM1_SCHPO'))
        self.assertEqual(record.dr_positive[1474], ('P08458', 'SPS1_YEAST'))
        self.assertEqual(record.dr_positive[1475], ('P13115', 'SRC1_XENLA'))
        self.assertEqual(record.dr_positive[1476], ('P13116', 'SRC2_XENLA'))
        self.assertEqual(record.dr_positive[1477], ('Q9V9J3', 'SRC42_DROME'))
        self.assertEqual(record.dr_positive[1478], ('P00528', 'SRC64_DROME'))
        self.assertEqual(record.dr_positive[1479], ('P15054', 'SRC_AVIS2'))
        self.assertEqual(record.dr_positive[1480], ('P00525', 'SRC_AVISR'))
        self.assertEqual(record.dr_positive[1481], ('P14084', 'SRC_AVISS'))
        self.assertEqual(record.dr_positive[1482], ('P14085', 'SRC_AVIST'))
        self.assertEqual(record.dr_positive[1483], ('P00523', 'SRC_CHICK'))
        self.assertEqual(record.dr_positive[1484], ('P12931', 'SRC_HUMAN'))
        self.assertEqual(record.dr_positive[1485], ('P05480', 'SRC_MOUSE'))
        self.assertEqual(record.dr_positive[1486], ('Q9WUD9', 'SRC_RAT'))
        self.assertEqual(record.dr_positive[1487], ('P25020', 'SRC_RSVH1'))
        self.assertEqual(record.dr_positive[1488], ('P00526', 'SRC_RSVP'))
        self.assertEqual(record.dr_positive[1489], ('P31693', 'SRC_RSVPA'))
        self.assertEqual(record.dr_positive[1490], ('P00524', 'SRC_RSVSA'))
        self.assertEqual(record.dr_positive[1491], ('P63185', 'SRC_RSVSE'))
        self.assertEqual(record.dr_positive[1492], ('P42686', 'SRK1_SPOLA'))
        self.assertEqual(record.dr_positive[1493], ('P42688', 'SRK2_SPOLA'))
        self.assertEqual(record.dr_positive[1494], ('P42689', 'SRK3_SPOLA'))
        self.assertEqual(record.dr_positive[1495], ('P42690', 'SRK4_SPOLA'))
        self.assertEqual(record.dr_positive[1496], ('Q09092', 'SRK6_BRAOE'))
        self.assertEqual(record.dr_positive[1497], ('Q9H3Y6', 'SRMS_HUMAN'))
        self.assertEqual(record.dr_positive[1498], ('Q62270', 'SRMS_MOUSE'))
        self.assertEqual(record.dr_positive[1499], ('Q96SB4', 'SRPK1_HUMAN'))

    def read1_positive3(self, record):
        self.assertEqual(len(record.dr_positive), 1689)
        self.assertEqual(record.dr_positive[1500], ('O70551', 'SRPK1_MOUSE'))
        self.assertEqual(record.dr_positive[1501], ('Q5RD27', 'SRPK1_PONPY'))
        self.assertEqual(record.dr_positive[1502], ('P78362', 'SRPK2_HUMAN'))
        self.assertEqual(record.dr_positive[1503], ('O54781', 'SRPK2_MOUSE'))
        self.assertEqual(record.dr_positive[1504], ('P25390', 'SSK22_YEAST'))
        self.assertEqual(record.dr_positive[1505], ('P53599', 'SSK2_YEAST'))
        self.assertEqual(record.dr_positive[1506], ('P50526', 'SSP1_SCHPO'))
        self.assertEqual(record.dr_positive[1507], ('Q9UEE5', 'ST17A_HUMAN'))
        self.assertEqual(record.dr_positive[1508], ('Q9GM70', 'ST17A_RABIT'))
        self.assertEqual(record.dr_positive[1509], ('O94768', 'ST17B_HUMAN'))
        self.assertEqual(record.dr_positive[1510], ('Q8BG48', 'ST17B_MOUSE'))
        self.assertEqual(record.dr_positive[1511], ('Q91XS8', 'ST17B_RAT'))
        self.assertEqual(record.dr_positive[1512], ('Q9Y2H1', 'ST38L_HUMAN'))
        self.assertEqual(record.dr_positive[1513], ('P23561', 'STE11_YEAST'))
        self.assertEqual(record.dr_positive[1514], ('Q92212', 'STE20_CANAL'))
        self.assertEqual(record.dr_positive[1515], ('Q03497', 'STE20_YEAST'))
        self.assertEqual(record.dr_positive[1516], ('P46599', 'STE7_CANAL'))
        self.assertEqual(record.dr_positive[1517], ('P06784', 'STE7_YEAST'))
        self.assertEqual(record.dr_positive[1518], ('O94804', 'STK10_HUMAN'))
        self.assertEqual(record.dr_positive[1519], ('O55098', 'STK10_MOUSE'))
        self.assertEqual(record.dr_positive[1520], ('Q15831', 'STK11_HUMAN'))
        self.assertEqual(record.dr_positive[1521], ('Q91604', 'STK11_XENLA'))
        self.assertEqual(record.dr_positive[1522], ('Q9UPE1', 'STK23_HUMAN'))
        self.assertEqual(record.dr_positive[1523], ('Q9Z0G2', 'STK23_MOUSE'))
        self.assertEqual(record.dr_positive[1524], ('Q9Y6E0', 'STK24_HUMAN'))
        self.assertEqual(record.dr_positive[1525], ('Q99KH8', 'STK24_MOUSE'))
        self.assertEqual(record.dr_positive[1526], ('O00506', 'STK25_HUMAN'))
        self.assertEqual(record.dr_positive[1527], ('Q9Z2W1', 'STK25_MOUSE'))
        self.assertEqual(record.dr_positive[1528], ('Q8TDR2', 'STK35_HUMAN'))
        self.assertEqual(record.dr_positive[1529], ('Q9NRP7', 'STK36_HUMAN'))
        self.assertEqual(record.dr_positive[1530], ('Q69ZM6', 'STK36_MOUSE'))
        self.assertEqual(record.dr_positive[1531], ('Q5RAJ5', 'STK36_PONPY'))
        self.assertEqual(record.dr_positive[1532], ('Q15208', 'STK38_HUMAN'))
        self.assertEqual(record.dr_positive[1533], ('Q91VJ4', 'STK38_MOUSE'))
        self.assertEqual(record.dr_positive[1534], ('Q9UEW8', 'STK39_HUMAN'))
        self.assertEqual(record.dr_positive[1535], ('Q9Z1W9', 'STK39_MOUSE'))
        self.assertEqual(record.dr_positive[1536], ('O88506', 'STK39_RAT'))
        self.assertEqual(record.dr_positive[1537], ('Q13188', 'STK3_HUMAN'))
        self.assertEqual(record.dr_positive[1538], ('Q9JI10', 'STK3_MOUSE'))
        self.assertEqual(record.dr_positive[1539], ('Q13043', 'STK4_HUMAN'))
        self.assertEqual(record.dr_positive[1540], ('Q91819', 'STK6L_XENLA'))
        self.assertEqual(record.dr_positive[1541], ('O14965', 'STK6_HUMAN'))
        self.assertEqual(record.dr_positive[1542], ('P97477', 'STK6_MOUSE'))
        self.assertEqual(record.dr_positive[1543], ('P59241', 'STK6_RAT'))
        self.assertEqual(record.dr_positive[1544], ('Q91820', 'STK6_XENLA'))
        self.assertEqual(record.dr_positive[1545], ('P17713', 'STK_HYDAT'))
        self.assertEqual(record.dr_positive[1546], ('Q9S713', 'STT7_ARATH'))
        self.assertEqual(record.dr_positive[1547], ('Q84V18', 'STT7_CHLRE'))
        self.assertEqual(record.dr_positive[1548], ('Q09892', 'STY1_SCHPO'))
        self.assertEqual(record.dr_positive[1549], ('P46549', 'SULU_CAEEL'))
        self.assertEqual(record.dr_positive[1550], ('P39745', 'SUR1_CAEEL'))
        self.assertEqual(record.dr_positive[1551], ('P32944', 'SWE1_YEAST'))
        self.assertEqual(record.dr_positive[1552], ('Q6NU21', 'TAO1A_XENLA'))
        self.assertEqual(record.dr_positive[1553], ('Q7ZYJ0', 'TAO1B_XENLA'))
        self.assertEqual(record.dr_positive[1554], ('Q7L7X3', 'TAOK1_HUMAN'))
        self.assertEqual(record.dr_positive[1555], ('Q5F2E8', 'TAOK1_MOUSE'))
        self.assertEqual(record.dr_positive[1556], ('O88664', 'TAOK1_RAT'))
        self.assertEqual(record.dr_positive[1557], ('Q9UL54', 'TAOK2_HUMAN'))
        self.assertEqual(record.dr_positive[1558], ('Q6ZQ29', 'TAOK2_MOUSE'))
        self.assertEqual(record.dr_positive[1559], ('Q9JLS3', 'TAOK2_RAT'))
        self.assertEqual(record.dr_positive[1560], ('Q6GPK9', 'TAOK2_XENLA'))
        self.assertEqual(record.dr_positive[1561], ('Q9I9E0', 'TAOK3_CHICK'))
        self.assertEqual(record.dr_positive[1562], ('Q9H2K8', 'TAOK3_HUMAN'))
        self.assertEqual(record.dr_positive[1563], ('Q8BYC6', 'TAOK3_MOUSE'))
        self.assertEqual(record.dr_positive[1564], ('Q5R4F3', 'TAOK3_PONPY'))
        self.assertEqual(record.dr_positive[1565], ('Q53UA7', 'TAOK3_RAT'))
        self.assertEqual(record.dr_positive[1566], ('Q6DD27', 'TAOK3_XENLA'))
        self.assertEqual(record.dr_positive[1567], ('Q9UHD2', 'TBK1_HUMAN'))
        self.assertEqual(record.dr_positive[1568], ('Q9WUN2', 'TBK1_MOUSE'))
        self.assertEqual(record.dr_positive[1569], ('Q6DFJ6', 'TBK1_XENLA'))
        self.assertEqual(record.dr_positive[1570], ('P42680', 'TEC_HUMAN'))
        self.assertEqual(record.dr_positive[1571], ('P24604', 'TEC_MOUSE'))
        self.assertEqual(record.dr_positive[1572], ('Q15569', 'TESK1_HUMAN'))
        self.assertEqual(record.dr_positive[1573], ('O70146', 'TESK1_MOUSE'))
        self.assertEqual(record.dr_positive[1574], ('Q63572', 'TESK1_RAT'))
        self.assertEqual(record.dr_positive[1575], ('Q96S53', 'TESK2_HUMAN'))
        self.assertEqual(record.dr_positive[1576], ('Q8VCT9', 'TESK2_MOUSE'))
        self.assertEqual(record.dr_positive[1577], ('Q924U5', 'TESK2_RAT'))
        self.assertEqual(record.dr_positive[1578], ('P36897', 'TGFR1_HUMAN'))
        self.assertEqual(record.dr_positive[1579], ('Q64729', 'TGFR1_MOUSE'))
        self.assertEqual(record.dr_positive[1580], ('P80204', 'TGFR1_RAT'))
        self.assertEqual(record.dr_positive[1581], ('P37173', 'TGFR2_HUMAN'))
        self.assertEqual(record.dr_positive[1582], ('Q62312', 'TGFR2_MOUSE'))
        self.assertEqual(record.dr_positive[1583], ('P38551', 'TGFR2_PIG'))
        self.assertEqual(record.dr_positive[1584], ('P38438', 'TGFR2_RAT'))
        self.assertEqual(record.dr_positive[1585], ('Q06805', 'TIE1_BOVIN'))
        self.assertEqual(record.dr_positive[1586], ('P35590', 'TIE1_HUMAN'))
        self.assertEqual(record.dr_positive[1587], ('Q06806', 'TIE1_MOUSE'))
        self.assertEqual(record.dr_positive[1588], ('Q06807', 'TIE2_BOVIN'))
        self.assertEqual(record.dr_positive[1589], ('O73791', 'TIE2_BRARE'))
        self.assertEqual(record.dr_positive[1590], ('Q02763', 'TIE2_HUMAN'))
        self.assertEqual(record.dr_positive[1591], ('Q02858', 'TIE2_MOUSE'))
        self.assertEqual(record.dr_positive[1592], ('P34314', 'TLK1_CAEEL'))
        self.assertEqual(record.dr_positive[1593], ('Q9UKI8', 'TLK1_HUMAN'))
        self.assertEqual(record.dr_positive[1594], ('Q8C0V0', 'TLK1_MOUSE'))
        self.assertEqual(record.dr_positive[1595], ('Q86UE8', 'TLK2_HUMAN'))
        self.assertEqual(record.dr_positive[1596], ('O55047', 'TLK2_MOUSE'))
        self.assertEqual(record.dr_positive[1597], ('P43298', 'TMK1_ARATH'))
        self.assertEqual(record.dr_positive[1598], ('Q59H18', 'TNI3K_HUMAN'))
        self.assertEqual(record.dr_positive[1599], ('Q5GIG6', 'TNI3K_MOUSE'))
        self.assertEqual(record.dr_positive[1600], ('Q5RF15', 'TNI3K_PONPY'))
        self.assertEqual(record.dr_positive[1601], ('Q7TQP6', 'TNI3K_RAT'))
        self.assertEqual(record.dr_positive[1602], ('Q9UKE5', 'TNIK_HUMAN'))
        self.assertEqual(record.dr_positive[1603], ('P83510', 'TNIK_MOUSE'))
        self.assertEqual(record.dr_positive[1604], ('Q13470', 'TNK1_HUMAN'))
        self.assertEqual(record.dr_positive[1605], ('Q99ML2', 'TNK1_MOUSE'))
        self.assertEqual(record.dr_positive[1606], ('P18475', 'TOR_DROME'))
        self.assertEqual(record.dr_positive[1607], ('O76997', 'TRK1_LYMST'))
        self.assertEqual(record.dr_positive[1608], ('Q9BXA7', 'TSSK1_HUMAN'))
        self.assertEqual(record.dr_positive[1609], ('Q61241', 'TSSK1_MOUSE'))
        self.assertEqual(record.dr_positive[1610], ('Q96PF2', 'TSSK2_HUMAN'))
        self.assertEqual(record.dr_positive[1611], ('O54863', 'TSSK2_MOUSE'))
        self.assertEqual(record.dr_positive[1612], ('Q96PN8', 'TSSK3_HUMAN'))
        self.assertEqual(record.dr_positive[1613], ('Q9D2E1', 'TSSK3_MOUSE'))
        self.assertEqual(record.dr_positive[1614], ('Q6SA08', 'TSSK4_HUMAN'))
        self.assertEqual(record.dr_positive[1615], ('Q9D411', 'TSSK4_MOUSE'))
        self.assertEqual(record.dr_positive[1616], ('Q9BXA6', 'TSSK6_HUMAN'))
        self.assertEqual(record.dr_positive[1617], ('Q925K9', 'TSSK6_MOUSE'))
        self.assertEqual(record.dr_positive[1618], ('P33981', 'TTK_HUMAN'))
        self.assertEqual(record.dr_positive[1619], ('P42681', 'TXK_HUMAN'))
        self.assertEqual(record.dr_positive[1620], ('P42682', 'TXK_MOUSE'))
        self.assertEqual(record.dr_positive[1621], ('P29597', 'TYK2_HUMAN'))
        self.assertEqual(record.dr_positive[1622], ('Q9R117', 'TYK2_MOUSE'))
        self.assertEqual(record.dr_positive[1623], ('Q06418', 'TYRO3_HUMAN'))
        self.assertEqual(record.dr_positive[1624], ('P55144', 'TYRO3_MOUSE'))
        self.assertEqual(record.dr_positive[1625], ('P55146', 'TYRO3_RAT'))
        self.assertEqual(record.dr_positive[1626], ('P30530', 'UFO_HUMAN'))
        self.assertEqual(record.dr_positive[1627], ('Q00993', 'UFO_MOUSE'))
        self.assertEqual(record.dr_positive[1628], ('O75385', 'ULK1_HUMAN'))
        self.assertEqual(record.dr_positive[1629], ('O70405', 'ULK1_MOUSE'))
        self.assertEqual(record.dr_positive[1630], ('Q8IYT8', 'ULK2_HUMAN'))
        self.assertEqual(record.dr_positive[1631], ('Q9QY01', 'ULK2_MOUSE'))
        self.assertEqual(record.dr_positive[1632], ('Q23023', 'UNC51_CAEEL'))
        self.assertEqual(record.dr_positive[1633], ('Q9J523', 'V212_FOWPV'))
        self.assertEqual(record.dr_positive[1634], ('Q9J509', 'V226_FOWPV'))
        self.assertEqual(record.dr_positive[1635], ('O61460', 'VAB1_CAEEL'))
        self.assertEqual(record.dr_positive[1636], ('P17948', 'VGFR1_HUMAN'))
        self.assertEqual(record.dr_positive[1637], ('P35969', 'VGFR1_MOUSE'))
        self.assertEqual(record.dr_positive[1638], ('P53767', 'VGFR1_RAT'))
        self.assertEqual(record.dr_positive[1639], ('Q8AXB3', 'VGFR2_BRARE'))
        self.assertEqual(record.dr_positive[1640], ('P52583', 'VGFR2_COTJA'))
        self.assertEqual(record.dr_positive[1641], ('P35968', 'VGFR2_HUMAN'))
        self.assertEqual(record.dr_positive[1642], ('P35918', 'VGFR2_MOUSE'))
        self.assertEqual(record.dr_positive[1643], ('O08775', 'VGFR2_RAT'))
        self.assertEqual(record.dr_positive[1644], ('P35916', 'VGFR3_HUMAN'))
        self.assertEqual(record.dr_positive[1645], ('P35917', 'VGFR3_MOUSE'))
        self.assertEqual(record.dr_positive[1646], ('Q91ZT1', 'VGFR3_RAT'))
        self.assertEqual(record.dr_positive[1647], ('Q03785', 'VHS1_YEAST'))
        self.assertEqual(record.dr_positive[1648], ('Q7ZUS1', 'VRK1_BRARE'))
        self.assertEqual(record.dr_positive[1649], ('Q99986', 'VRK1_HUMAN'))
        self.assertEqual(record.dr_positive[1650], ('Q80X41', 'VRK1_MOUSE'))
        self.assertEqual(record.dr_positive[1651], ('P47817', 'WEE1A_XENLA'))
        self.assertEqual(record.dr_positive[1652], ('P54350', 'WEE1_DROME'))
        self.assertEqual(record.dr_positive[1653], ('P30291', 'WEE1_HUMAN'))
        self.assertEqual(record.dr_positive[1654], ('P47810', 'WEE1_MOUSE'))
        self.assertEqual(record.dr_positive[1655], ('Q63802', 'WEE1_RAT'))
        self.assertEqual(record.dr_positive[1656], ('P07527', 'WEE1_SCHPO'))
        self.assertEqual(record.dr_positive[1657], ('O74304', 'WIN1_SCHPO'))
        self.assertEqual(record.dr_positive[1658], ('P33886', 'WIS1_SCHPO'))
        self.assertEqual(record.dr_positive[1659], ('P13388', 'XMRK_XIPMA'))
        self.assertEqual(record.dr_positive[1660], ('Q8MYQ1', 'Y31E_CAEEL'))
        self.assertEqual(record.dr_positive[1661], ('P14680', 'YAK1_YEAST'))
        self.assertEqual(record.dr_positive[1662], ('P00527', 'YES_AVISY'))
        self.assertEqual(record.dr_positive[1663], ('Q28923', 'YES_CANFA'))
        self.assertEqual(record.dr_positive[1664], ('P09324', 'YES_CHICK'))
        self.assertEqual(record.dr_positive[1665], ('P07947', 'YES_HUMAN'))
        self.assertEqual(record.dr_positive[1666], ('Q04736', 'YES_MOUSE'))
        self.assertEqual(record.dr_positive[1667], ('P10936', 'YES_XENLA'))
        self.assertEqual(record.dr_positive[1668], ('P27447', 'YES_XIPHE'))
        self.assertEqual(record.dr_positive[1669], ('P42168', 'YKL1_CAEEL'))
        self.assertEqual(record.dr_positive[1670], ('P42169', 'YKL2_CAEEL'))
        self.assertEqual(record.dr_positive[1671], ('Q5UQ24', 'YL205_MIMIV'))
        self.assertEqual(record.dr_positive[1672], ('Q5UNT4', 'YL670_MIMIV'))
        self.assertEqual(record.dr_positive[1673], ('Q5UNT1', 'YL673_MIMIV'))
        self.assertEqual(record.dr_positive[1674], ('P32742', 'YNH4_CAEEL'))
        self.assertEqual(record.dr_positive[1675], ('P34635', 'YOO3_CAEEL'))
        self.assertEqual(record.dr_positive[1676], ('Q11179', 'YPC2_CAEEL'))
        self.assertEqual(record.dr_positive[1677], ('P12688', 'YPK1_YEAST'))
        self.assertEqual(record.dr_positive[1678], ('P18961', 'YPK2_YEAST'))
        self.assertEqual(record.dr_positive[1679], ('Q20085', 'YPS7_CAEEL'))
        self.assertEqual(record.dr_positive[1680], ('Q20347', 'YR62_CAEEL'))
        self.assertEqual(record.dr_positive[1681], ('Q7T6Y2', 'YR831_MIMIV'))
        self.assertEqual(record.dr_positive[1682], ('Q02977', 'YRK_CHICK'))
        self.assertEqual(record.dr_positive[1683], ('Q11090', 'YWY3_CAEEL'))
        self.assertEqual(record.dr_positive[1684], ('Q11112', 'YX05_CAEEL'))
        self.assertEqual(record.dr_positive[1685], ('P43403', 'ZAP70_HUMAN'))
        self.assertEqual(record.dr_positive[1686], ('P43404', 'ZAP70_MOUSE'))
        self.assertEqual(record.dr_positive[1687], ('Q621J7', 'ZYG1_CAEBR'))
        self.assertEqual(record.dr_positive[1688], ('Q9GT24', 'ZYG1_CAEEL'))

    def read1_false_neg(self, record):
        self.assertEqual(len(record.dr_false_neg), 324)
        self.assertEqual(record.dr_false_neg[0], ('P51566', 'AFC1_ARATH'))
        self.assertEqual(record.dr_false_neg[1], ('P51567', 'AFC2_ARATH'))
        self.assertEqual(record.dr_false_neg[2], ('P51568', 'AFC3_ARATH'))
        self.assertEqual(record.dr_false_neg[3], ('P38080', 'AKL1_YEAST'))
        self.assertEqual(record.dr_false_neg[4], ('Q9C0K7', 'AL2S2_HUMAN'))
        self.assertEqual(record.dr_false_neg[5], ('Q8K4T3', 'AL2S2_MOUSE'))
        self.assertEqual(record.dr_false_neg[6], ('Q16671', 'AMHR2_HUMAN'))
        self.assertEqual(record.dr_false_neg[7], ('Q62893', 'AMHR2_RAT'))
        self.assertEqual(record.dr_false_neg[8], ('Q8NFD2', 'ANKK1_HUMAN'))
        self.assertEqual(record.dr_false_neg[9], ('Q8BZ25', 'ANKK1_MOUSE'))
        self.assertEqual(record.dr_false_neg[10], ('P53974', 'ARK1_YEAST'))
        self.assertEqual(record.dr_false_neg[11], ('Q28043', 'AVR2A_BOVIN'))
        self.assertEqual(record.dr_false_neg[12], ('P27037', 'AVR2A_HUMAN'))
        self.assertEqual(record.dr_false_neg[13], ('P27038', 'AVR2A_MOUSE'))
        self.assertEqual(record.dr_false_neg[14], ('P38444', 'AVR2A_RAT'))
        self.assertEqual(record.dr_false_neg[15], ('Q28560', 'AVR2A_SHEEP'))
        self.assertEqual(record.dr_false_neg[16], ('P27039', 'AVR2A_XENLA'))
        self.assertEqual(record.dr_false_neg[17], ('Q95126', 'AVR2B_BOVIN'))
        self.assertEqual(record.dr_false_neg[18], ('Q13705', 'AVR2B_HUMAN'))
        self.assertEqual(record.dr_false_neg[19], ('P27040', 'AVR2B_MOUSE'))
        self.assertEqual(record.dr_false_neg[20], ('P38445', 'AVR2B_RAT'))
        self.assertEqual(record.dr_false_neg[21], ('P27041', 'AVR2B_XENLA'))
        self.assertEqual(record.dr_false_neg[22], ('Q9NSY1', 'BMP2K_HUMAN'))
        self.assertEqual(record.dr_false_neg[23], ('Q91Z96', 'BMP2K_MOUSE'))
        self.assertEqual(record.dr_false_neg[24], ('O60566', 'BUB1B_HUMAN'))
        self.assertEqual(record.dr_false_neg[25], ('Q9Z1S0', 'BUB1B_MOUSE'))
        self.assertEqual(record.dr_false_neg[26], ('P53323', 'BUD32_YEAST'))
        self.assertEqual(record.dr_false_neg[27], ('P43568', 'CAK1_YEAST'))
        self.assertEqual(record.dr_false_neg[28], ('Q24210', 'CAKI_DROME'))
        self.assertEqual(record.dr_false_neg[29], ('Q6RET6', 'CCAMK_PEA'))
        self.assertEqual(record.dr_false_neg[30], ('P21127', 'CD2L1_HUMAN'))
        self.assertEqual(record.dr_false_neg[31], ('P24788', 'CD2L1_MOUSE'))
        self.assertEqual(record.dr_false_neg[32], ('P46892', 'CD2L1_RAT'))
        self.assertEqual(record.dr_false_neg[33], ('Q9UQ88', 'CD2L2_HUMAN'))
        self.assertEqual(record.dr_false_neg[34], ('Q5RCH1', 'CDC2_PONPY'))
        self.assertEqual(record.dr_false_neg[35], ('P06243', 'CDC7_YEAST'))
        self.assertEqual(record.dr_false_neg[36], ('Q5MAI5', 'CDKL4_HUMAN'))
        self.assertEqual(record.dr_false_neg[37], ('P87050', 'CDR2_SCHPO'))
        self.assertEqual(record.dr_false_neg[38], ('P38938', 'CEK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[39], ('O96017', 'CHK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[40], ('Q9Z265', 'CHK2_MOUSE'))
        self.assertEqual(record.dr_false_neg[41], ('Q8NE28', 'CI096_HUMAN'))
        self.assertEqual(record.dr_false_neg[42], ('Q4R793', 'CI096_MACFA'))
        self.assertEqual(record.dr_false_neg[43], ('Q9HFW2', 'CLA4_ASHGO'))
        self.assertEqual(record.dr_false_neg[44], ('O14427', 'CLA4_CANAL'))
        self.assertEqual(record.dr_false_neg[45], ('P48562', 'CLA4_YEAST'))
        self.assertEqual(record.dr_false_neg[46], ('Q9SYQ8', 'CLV1_ARATH'))
        self.assertEqual(record.dr_false_neg[47], ('P36615', 'CSK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[48], ('O14936', 'CSKP_HUMAN'))
        self.assertEqual(record.dr_false_neg[49], ('O70589', 'CSKP_MOUSE'))
        self.assertEqual(record.dr_false_neg[50], ('Q62915', 'CSKP_RAT'))
        self.assertEqual(record.dr_false_neg[51], ('P50488', 'DAF4_CAEEL'))
        self.assertEqual(record.dr_false_neg[52], ('Q08345', 'DDR1_HUMAN'))
        self.assertEqual(record.dr_false_neg[53], ('Q03146', 'DDR1_MOUSE'))
        self.assertEqual(record.dr_false_neg[54], ('Q7YR43', 'DDR1_PANTR'))
        self.assertEqual(record.dr_false_neg[55], ('Q9Z2B5', 'E2AK3_MOUSE'))
        self.assertEqual(record.dr_false_neg[56], ('Q9Z1Z1', 'E2AK3_RAT'))
        self.assertEqual(record.dr_false_neg[57], ('P32801', 'ELM1_YEAST'))
        self.assertEqual(record.dr_false_neg[58], ('P54757', 'EPHA5_RAT'))
        self.assertEqual(record.dr_false_neg[59], ('Q5JZY3', 'EPHAA_HUMAN'))
        self.assertEqual(record.dr_false_neg[60], ('P21860', 'ERBB3_HUMAN'))
        self.assertEqual(record.dr_false_neg[61], ('Q61526', 'ERBB3_MOUSE'))
        self.assertEqual(record.dr_false_neg[62], ('Q5RB22', 'ERBB3_PONPY'))
        self.assertEqual(record.dr_false_neg[63], ('Q62799', 'ERBB3_RAT'))
        self.assertEqual(record.dr_false_neg[64], ('O75460', 'ERN1_HUMAN'))
        self.assertEqual(record.dr_false_neg[65], ('Q9EQY0', 'ERN1_MOUSE'))
        self.assertEqual(record.dr_false_neg[66], ('Q76MJ5', 'ERN2_HUMAN'))
        self.assertEqual(record.dr_false_neg[67], ('Q9Z2E3', 'ERN2_MOUSE'))
        self.assertEqual(record.dr_false_neg[68], ('O13839', 'FIN1_SCHPO'))
        self.assertEqual(record.dr_false_neg[69], ('P47997', 'G11A_ORYSA'))
        self.assertEqual(record.dr_false_neg[70], ('O14976', 'GAK_HUMAN'))
        self.assertEqual(record.dr_false_neg[71], ('Q99KY4', 'GAK_MOUSE'))
        self.assertEqual(record.dr_false_neg[72], ('P97874', 'GAK_RAT'))
        self.assertEqual(record.dr_false_neg[73], ('P24446', 'GCVK_HHV6U'))
        self.assertEqual(record.dr_false_neg[74], ('P52446', 'GCVK_HHV6Z'))
        self.assertEqual(record.dr_false_neg[75], ('P52344', 'GCVK_HHV7J'))
        self.assertEqual(record.dr_false_neg[76], ('Q01015', 'GCVK_SHV21'))
        self.assertEqual(record.dr_false_neg[77], ('P38970', 'HAL5_YEAST'))
        self.assertEqual(record.dr_false_neg[78], ('P50582', 'HSK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[79], ('Q9VEZ5', 'IKKB_DROME'))
        self.assertEqual(record.dr_false_neg[80], ('O14920', 'IKKB_HUMAN'))
        self.assertEqual(record.dr_false_neg[81], ('O88351', 'IKKB_MOUSE'))
        self.assertEqual(record.dr_false_neg[82], ('Q9QY78', 'IKKB_RAT'))
        self.assertEqual(record.dr_false_neg[83], ('Q13418', 'ILK1_HUMAN'))
        self.assertEqual(record.dr_false_neg[84], ('P57043', 'ILK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[85], ('P57044', 'ILK_CAVPO'))
        self.assertEqual(record.dr_false_neg[86], ('O55222', 'ILK_MOUSE'))
        self.assertEqual(record.dr_false_neg[87], ('P32581', 'IME2_YEAST'))
        self.assertEqual(record.dr_false_neg[88], ('Q59S66', 'IPL1_CANAL'))
        self.assertEqual(record.dr_false_neg[89], ('Q6BVA0', 'IPL1_DEBHA'))
        self.assertEqual(record.dr_false_neg[90], ('O43187', 'IRAK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[91], ('Q9Y616', 'IRAK3_HUMAN'))
        self.assertEqual(record.dr_false_neg[92], ('Q8K4B2', 'IRAK3_MOUSE'))
        self.assertEqual(record.dr_false_neg[93], ('Q9NWZ3', 'IRAK4_HUMAN'))
        self.assertEqual(record.dr_false_neg[94], ('Q8R4K2', 'IRAK4_MOUSE'))
        self.assertEqual(record.dr_false_neg[95], ('P32361', 'IRE1_YEAST'))
        self.assertEqual(record.dr_false_neg[96], ('Q758T2', 'ISR1_ASHGO'))
        self.assertEqual(record.dr_false_neg[97], ('Q9U6D2', 'JNK1_ANCCA'))
        self.assertEqual(record.dr_false_neg[98], ('Q8WQG9', 'JNK1_CAEEL'))
        self.assertEqual(record.dr_false_neg[99], ('P92208', 'JNK_DROME'))
        self.assertEqual(record.dr_false_neg[100], ('Q966Y3', 'JNK_SUBDO'))
        self.assertEqual(record.dr_false_neg[101], ('P31374', 'KAB7_YEAST'))
        self.assertEqual(record.dr_false_neg[102], ('Q09831', 'KAD5_SCHPO'))
        self.assertEqual(record.dr_false_neg[103], ('P39962', 'KC13_YEAST'))
        self.assertEqual(record.dr_false_neg[104], ('Q9UU87', 'KC61_SCHPO'))
        self.assertEqual(record.dr_false_neg[105], ('P53233', 'KG1Z_YEAST'))
        self.assertEqual(record.dr_false_neg[106], ('P25341', 'KIN82_YEAST'))
        self.assertEqual(record.dr_false_neg[107], ('O74526', 'KJ45_SCHPO'))
        self.assertEqual(record.dr_false_neg[108], ('P47042', 'KJF7_YEAST'))
        self.assertEqual(record.dr_false_neg[109], ('P36004', 'KKQ8_YEAST'))
        self.assertEqual(record.dr_false_neg[110], ('P53739', 'KN8R_YEAST'))
        self.assertEqual(record.dr_false_neg[111], ('Q08217', 'KOE5_YEAST'))
        self.assertEqual(record.dr_false_neg[112], ('Q9VPC0', 'KP58_DROME'))
        self.assertEqual(record.dr_false_neg[113], ('P15792', 'KPK1_PHAVU'))
        self.assertEqual(record.dr_false_neg[114], ('Q05999', 'KPK7_ARATH'))
        self.assertEqual(record.dr_false_neg[115], ('P15443', 'KR14_ICHV1'))
        self.assertEqual(record.dr_false_neg[116], ('Q00097', 'KR15_ICHV1'))
        self.assertEqual(record.dr_false_neg[117], ('Q00098', 'KR16_ICHV1'))
        self.assertEqual(record.dr_false_neg[118], ('Q08097', 'KR1_BHV1S'))
        self.assertEqual(record.dr_false_neg[119], ('Q04543', 'KR1_CHV9D'))
        self.assertEqual(record.dr_false_neg[120], ('P28926', 'KR1_EHV1B'))
        self.assertEqual(record.dr_false_neg[121], ('P32516', 'KR1_EHV1K'))
        self.assertEqual(record.dr_false_neg[122], ('P84390', 'KR1_EHV1V'))
        self.assertEqual(record.dr_false_neg[123], ('P04413', 'KR1_HHV11'))
        self.assertEqual(record.dr_false_neg[124], ('P13287', 'KR1_HHV2H'))
        self.assertEqual(record.dr_false_neg[125], ('P17613', 'KR1_PRVKA'))
        self.assertEqual(record.dr_false_neg[126], ('P24381', 'KR1_PRVN3'))
        self.assertEqual(record.dr_false_neg[127], ('P09251', 'KR1_VZVD'))
        self.assertEqual(record.dr_false_neg[128], ('P13288', 'KR2_EBV'))
        self.assertEqual(record.dr_false_neg[129], ('P28966', 'KR2_EHV1B'))
        self.assertEqual(record.dr_false_neg[130], ('P84391', 'KR2_EHV1V'))
        self.assertEqual(record.dr_false_neg[131], ('P04290', 'KR2_HHV11'))
        self.assertEqual(record.dr_false_neg[132], ('P30662', 'KR2_PRVN3'))
        self.assertEqual(record.dr_false_neg[133], ('P09296', 'KR2_VZVD'))
        self.assertEqual(record.dr_false_neg[134], ('Q00094', 'KR73_ICHV1'))
        self.assertEqual(record.dr_false_neg[135], ('Q00095', 'KR74_ICHV1'))
        self.assertEqual(record.dr_false_neg[136], ('O57259', 'KRB2_VACCA'))
        self.assertEqual(record.dr_false_neg[137], ('P21098', 'KRB2_VACCC'))
        self.assertEqual(record.dr_false_neg[138], ('P24362', 'KRB2_VACCV'))
        self.assertEqual(record.dr_false_neg[139], ('P38691', 'KSP1_YEAST'))
        self.assertEqual(record.dr_false_neg[140], ('O95835', 'LATS1_HUMAN'))
        self.assertEqual(record.dr_false_neg[141], ('Q8BYR2', 'LATS1_MOUSE'))
        self.assertEqual(record.dr_false_neg[142], ('Q12852', 'M3K12_HUMAN'))
        self.assertEqual(record.dr_false_neg[143], ('Q60700', 'M3K12_MOUSE'))
        self.assertEqual(record.dr_false_neg[144], ('Q63796', 'M3K12_RAT'))
        self.assertEqual(record.dr_false_neg[145], ('O43283', 'M3K13_HUMAN'))
        self.assertEqual(record.dr_false_neg[146], ('Q5R8X7', 'M3K13_PONPY'))
        self.assertEqual(record.dr_false_neg[147], ('Q99759', 'M3K3_HUMAN'))
        self.assertEqual(record.dr_false_neg[148], ('Q61084', 'M3K3_MOUSE'))
        self.assertEqual(record.dr_false_neg[149], ('P41279', 'M3K8_HUMAN'))
        self.assertEqual(record.dr_false_neg[150], ('Q07174', 'M3K8_MOUSE'))
        self.assertEqual(record.dr_false_neg[151], ('Q63562', 'M3K8_RAT'))
        self.assertEqual(record.dr_false_neg[152], ('Q95UN8', 'M3KSL_DROME'))
        self.assertEqual(record.dr_false_neg[153], ('O14002', 'MAK2_SCHPO'))
        self.assertEqual(record.dr_false_neg[154], ('O74539', 'MAK3_SCHPO'))
        self.assertEqual(record.dr_false_neg[155], ('Q8IW41', 'MAPK5_HUMAN'))
        self.assertEqual(record.dr_false_neg[156], ('O54992', 'MAPK5_MOUSE'))
        self.assertEqual(record.dr_false_neg[157], ('Q9Y2H9', 'MAST1_HUMAN'))
        self.assertEqual(record.dr_false_neg[158], ('Q9R1L5', 'MAST1_MOUSE'))
        self.assertEqual(record.dr_false_neg[159], ('Q810W7', 'MAST1_RAT'))
        self.assertEqual(record.dr_false_neg[160], ('Q6P0Q8', 'MAST2_HUMAN'))
        self.assertEqual(record.dr_false_neg[161], ('Q60592', 'MAST2_MOUSE'))
        self.assertEqual(record.dr_false_neg[162], ('O60307', 'MAST3_HUMAN'))
        self.assertEqual(record.dr_false_neg[163], ('Q96GX5', 'MASTL_HUMAN'))
        self.assertEqual(record.dr_false_neg[164], ('Q8C0P0', 'MASTL_MOUSE'))
        self.assertEqual(record.dr_false_neg[165], ('P43294', 'MHK_ARATH'))
        self.assertEqual(record.dr_false_neg[166], ('P30290', 'MIK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[167], ('Q90327', 'MK08A_CYPCA'))
        self.assertEqual(record.dr_false_neg[168], ('O42099', 'MK08B_CYPCA'))
        self.assertEqual(record.dr_false_neg[169], ('Q9DGD9', 'MK08_BRARE'))
        self.assertEqual(record.dr_false_neg[170], ('P45983', 'MK08_HUMAN'))
        self.assertEqual(record.dr_false_neg[171], ('Q91Y86', 'MK08_MOUSE'))
        self.assertEqual(record.dr_false_neg[172], ('P49185', 'MK08_RAT'))
        self.assertEqual(record.dr_false_neg[173], ('Q8QHK8', 'MK08_XENLA'))
        self.assertEqual(record.dr_false_neg[174], ('P79996', 'MK09_CHICK'))
        self.assertEqual(record.dr_false_neg[175], ('P45984', 'MK09_HUMAN'))
        self.assertEqual(record.dr_false_neg[176], ('Q9WTU6', 'MK09_MOUSE'))
        self.assertEqual(record.dr_false_neg[177], ('P49186', 'MK09_RAT'))
        self.assertEqual(record.dr_false_neg[178], ('P53779', 'MK10_HUMAN'))
        self.assertEqual(record.dr_false_neg[179], ('Q61831', 'MK10_MOUSE'))
        self.assertEqual(record.dr_false_neg[180], ('P49187', 'MK10_RAT'))
        self.assertEqual(record.dr_false_neg[181], ('P43068', 'MKC1_CANAL'))
        self.assertEqual(record.dr_false_neg[182], ('Q9NYL2', 'MLTK_HUMAN'))
        self.assertEqual(record.dr_false_neg[183], ('Q9ESL4', 'MLTK_MOUSE'))
        self.assertEqual(record.dr_false_neg[184], ('O14733', 'MP2K7_HUMAN'))
        self.assertEqual(record.dr_false_neg[185], ('Q8NG66', 'NEK11_HUMAN'))
        self.assertEqual(record.dr_false_neg[186], ('Q8WNU8', 'NEK11_MACFA'))
        self.assertEqual(record.dr_false_neg[187], ('Q8C0Q4', 'NEK11_MOUSE'))
        self.assertEqual(record.dr_false_neg[188], ('P51955', 'NEK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[189], ('O35942', 'NEK2_MOUSE'))
        self.assertEqual(record.dr_false_neg[190], ('Q8TD19', 'NEK9_HUMAN'))
        self.assertEqual(record.dr_false_neg[191], ('Q8K1R7', 'NEK9_MOUSE'))
        self.assertEqual(record.dr_false_neg[192], ('Q7ZZC8', 'NEK9_XENLA'))
        self.assertEqual(record.dr_false_neg[193], ('P48479', 'NIM1_NEUCR'))
        self.assertEqual(record.dr_false_neg[194], ('P11837', 'NIMA_EMENI'))
        self.assertEqual(record.dr_false_neg[195], ('P10676', 'NINAC_DROME'))
        self.assertEqual(record.dr_false_neg[196], ('O42626', 'NRC2_NEUCR'))
        self.assertEqual(record.dr_false_neg[197], ('Q03428', 'NRKB_TRYBB'))
        self.assertEqual(record.dr_false_neg[198], ('P08018', 'PBS2_YEAST'))
        self.assertEqual(record.dr_false_neg[199], ('Q32PP3', 'PDK1L_BRARE'))
        self.assertEqual(record.dr_false_neg[200], ('Q9Y1J3', 'PDPK1_CAEEL'))
        self.assertEqual(record.dr_false_neg[201], ('Q99570', 'PI3R4_HUMAN'))
        self.assertEqual(record.dr_false_neg[202], ('Q8VD65', 'PI3R4_MOUSE'))
        self.assertEqual(record.dr_false_neg[203], ('Q5R9I3', 'PI3R4_PONPY'))
        self.assertEqual(record.dr_false_neg[204], ('P0C0R5', 'PI3R4_RAT'))
        self.assertEqual(record.dr_false_neg[205], ('Q91822', 'PIM3_XENLA'))
        self.assertEqual(record.dr_false_neg[206], ('Q9BXM7', 'PINK1_HUMAN'))
        self.assertEqual(record.dr_false_neg[207], ('P42493', 'PK1_ASFB7'))
        self.assertEqual(record.dr_false_neg[208], ('P34206', 'PK1_ASFM2'))
        self.assertEqual(record.dr_false_neg[209], ('P41415', 'PK1_NPVAC'))
        self.assertEqual(record.dr_false_neg[210], ('P41719', 'PK1_NPVHZ'))
        self.assertEqual(record.dr_false_neg[211], ('P41720', 'PK1_NPVLD'))
        self.assertEqual(record.dr_false_neg[212], ('O10269', 'PK1_NPVOP'))
        self.assertEqual(record.dr_false_neg[213], ('Q9KIG4', 'PK1_STRTO'))
        self.assertEqual(record.dr_false_neg[214], ('P41676', 'PK2_NPVAC'))
        self.assertEqual(record.dr_false_neg[215], ('P54739', 'PKAA_STRCO'))
        self.assertEqual(record.dr_false_neg[216], ('P54740', 'PKAB_STRCO'))
        self.assertEqual(record.dr_false_neg[217], ('P34100', 'PKD1_DICDI'))
        self.assertEqual(record.dr_false_neg[218], ('P54737', 'PKN5_MYXXA'))
        self.assertEqual(record.dr_false_neg[219], ('P54738', 'PKN6_MYXXA'))
        self.assertEqual(record.dr_false_neg[220], ('Q8G4G1', 'PKNA2_BIFLO'))
        self.assertEqual(record.dr_false_neg[221], ('P65727', 'PKNA_MYCBO'))
        self.assertEqual(record.dr_false_neg[222], ('P54743', 'PKNA_MYCLE'))
        self.assertEqual(record.dr_false_neg[223], ('P65726', 'PKNA_MYCTU'))
        self.assertEqual(record.dr_false_neg[224], ('Q7TZN1', 'PKNF_MYCBO'))
        self.assertEqual(record.dr_false_neg[225], ('P72003', 'PKNF_MYCTU'))
        self.assertEqual(record.dr_false_neg[226], ('P65729', 'PKNG_MYCBO'))
        self.assertEqual(record.dr_false_neg[227], ('P57993', 'PKNG_MYCLE'))
        self.assertEqual(record.dr_false_neg[228], ('P65728', 'PKNG_MYCTU'))
        self.assertEqual(record.dr_false_neg[229], ('P65731', 'PKNI_MYCBO'))
        self.assertEqual(record.dr_false_neg[230], ('P65730', 'PKNI_MYCTU'))
        self.assertEqual(record.dr_false_neg[231], ('P65733', 'PKNJ_MYCBO'))
        self.assertEqual(record.dr_false_neg[232], ('P65732', 'PKNJ_MYCTU'))
        self.assertEqual(record.dr_false_neg[233], ('Q7TYY6', 'PKNL_MYCBO'))
        self.assertEqual(record.dr_false_neg[234], ('O53510', 'PKNL_MYCTU'))
        self.assertEqual(record.dr_false_neg[235], ('P47355', 'PKNS_MYCGE'))
        self.assertEqual(record.dr_false_neg[236], ('P75524', 'PKNS_MYCPN'))
        self.assertEqual(record.dr_false_neg[237], ('Q01577', 'PKPA_PHYBL'))
        self.assertEqual(record.dr_false_neg[238], ('P52304', 'POLO_DROME'))
        self.assertEqual(record.dr_false_neg[239], ('O13958', 'PRK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[240], ('P40494', 'PRK1_YEAST'))
        self.assertEqual(record.dr_false_neg[241], ('O43930', 'PRKY_HUMAN'))
        self.assertEqual(record.dr_false_neg[242], ('Q13523', 'PRP4B_HUMAN'))
        self.assertEqual(record.dr_false_neg[243], ('Q61136', 'PRP4B_MOUSE'))
        self.assertEqual(record.dr_false_neg[244], ('Q96S44', 'PRPK_HUMAN'))
        self.assertEqual(record.dr_false_neg[245], ('Q99PW4', 'PRPK_MOUSE'))
        self.assertEqual(record.dr_false_neg[246], ('Q12706', 'PSK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[247], ('P36002', 'PTK1_YEAST'))
        self.assertEqual(record.dr_false_neg[248], ('P43565', 'RIM15_YEAST'))
        self.assertEqual(record.dr_false_neg[249], ('O43353', 'RIPK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[250], ('P58801', 'RIPK2_MOUSE'))
        self.assertEqual(record.dr_false_neg[251], ('Q9Y572', 'RIPK3_HUMAN'))
        self.assertEqual(record.dr_false_neg[252], ('Q01973', 'ROR1_HUMAN'))
        self.assertEqual(record.dr_false_neg[253], ('Q9Z139', 'ROR1_MOUSE'))
        self.assertEqual(record.dr_false_neg[254], ('Q01974', 'ROR2_HUMAN'))
        self.assertEqual(record.dr_false_neg[255], ('Q9Z138', 'ROR2_MOUSE'))
        self.assertEqual(record.dr_false_neg[256], ('P42159', 'RTK2_GEOCY'))
        self.assertEqual(record.dr_false_neg[257], ('Q27324', 'RYK1_DROME'))
        self.assertEqual(record.dr_false_neg[258], ('Q9V422', 'RYK2_DROME'))
        self.assertEqual(record.dr_false_neg[259], ('P34925', 'RYK_HUMAN'))
        self.assertEqual(record.dr_false_neg[260], ('Q01887', 'RYK_MOUSE'))
        self.assertEqual(record.dr_false_neg[261], ('P73515', 'SPKE_SYNY3'))
        self.assertEqual(record.dr_false_neg[262], ('Q9UQY9', 'SPO4_SCHPO'))
        self.assertEqual(record.dr_false_neg[263], ('O94547', 'SRK1_SCHPO'))
        self.assertEqual(record.dr_false_neg[264], ('Q7TSE6', 'ST38L_MOUSE'))
        self.assertEqual(record.dr_false_neg[265], ('O75716', 'STK16_HUMAN'))
        self.assertEqual(record.dr_false_neg[266], ('O88697', 'STK16_MOUSE'))
        self.assertEqual(record.dr_false_neg[267], ('P57760', 'STK16_RAT'))
        self.assertEqual(record.dr_false_neg[268], ('Q9BXU1', 'STK31_HUMAN'))
        self.assertEqual(record.dr_false_neg[269], ('P83098', 'STLK_DROME'))
        self.assertEqual(record.dr_false_neg[270], ('Q6J9G0', 'STYK1_HUMAN'))
        self.assertEqual(record.dr_false_neg[271], ('Q6J9G1', 'STYK1_MOUSE'))
        self.assertEqual(record.dr_false_neg[272], ('Q6DHU8', 'TOPK_BRARE'))
        self.assertEqual(record.dr_false_neg[273], ('Q96KB5', 'TOPK_HUMAN'))
        self.assertEqual(record.dr_false_neg[274], ('Q9JJ78', 'TOPK_MOUSE'))
        self.assertEqual(record.dr_false_neg[275], ('O75962', 'TRIO_HUMAN'))
        self.assertEqual(record.dr_false_neg[276], ('P35761', 'TTK_MOUSE'))
        self.assertEqual(record.dr_false_neg[277], ('Q05101', 'U1206_GAHVG'))
        self.assertEqual(record.dr_false_neg[278], ('Q8TAS1', 'UHMK1_HUMAN'))
        self.assertEqual(record.dr_false_neg[279], ('P97343', 'UHMK1_MOUSE'))
        self.assertEqual(record.dr_false_neg[280], ('Q63285', 'UHMK1_RAT'))
        self.assertEqual(record.dr_false_neg[281], ('P39073', 'UME5_YEAST'))
        self.assertEqual(record.dr_false_neg[282], ('O01761', 'UNC89_CAEEL'))
        self.assertEqual(record.dr_false_neg[283], ('Q9J5B1', 'V111_FOWPV'))
        self.assertEqual(record.dr_false_neg[284], ('O57252', 'VPK1_VACCA'))
        self.assertEqual(record.dr_false_neg[285], ('P20505', 'VPK1_VACCC'))
        self.assertEqual(record.dr_false_neg[286], ('P16913', 'VPK1_VACCV'))
        self.assertEqual(record.dr_false_neg[287], ('P33800', 'VPK1_VARV'))
        self.assertEqual(record.dr_false_neg[288], ('P32216', 'VPK2_SWPVK'))
        self.assertEqual(record.dr_false_neg[289], ('O57177', 'VPK2_VACCA'))
        self.assertEqual(record.dr_false_neg[290], ('P21095', 'VPK2_VACCC'))
        self.assertEqual(record.dr_false_neg[291], ('P29884', 'VPK2_VACCP'))
        self.assertEqual(record.dr_false_neg[292], ('Q9JFE5', 'VPK2_VACCT'))
        self.assertEqual(record.dr_false_neg[293], ('Q89121', 'VPK2_VACCV'))
        self.assertEqual(record.dr_false_neg[294], ('P33801', 'VPK2_VARV'))
        self.assertEqual(record.dr_false_neg[295], ('Q9UVG6', 'VPS15_PICPA'))
        self.assertEqual(record.dr_false_neg[296], ('P22219', 'VPS15_YEAST'))
        self.assertEqual(record.dr_false_neg[297], ('Q86Y07', 'VRK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[298], ('Q8BN21', 'VRK2_MOUSE'))
        self.assertEqual(record.dr_false_neg[299], ('Q8IV63', 'VRK3_HUMAN'))
        self.assertEqual(record.dr_false_neg[300], ('Q8K3G5', 'VRK3_MOUSE'))
        self.assertEqual(record.dr_false_neg[301], ('O14299', 'WIS4_SCHPO'))
        self.assertEqual(record.dr_false_neg[302], ('Q9H4A3', 'WNK1_HUMAN'))
        self.assertEqual(record.dr_false_neg[303], ('P83741', 'WNK1_MOUSE'))
        self.assertEqual(record.dr_false_neg[304], ('Q9JIH7', 'WNK1_RAT'))
        self.assertEqual(record.dr_false_neg[305], ('Q9Y3S1', 'WNK2_HUMAN'))
        self.assertEqual(record.dr_false_neg[306], ('Q9BYP7', 'WNK3_HUMAN'))
        self.assertEqual(record.dr_false_neg[307], ('Q96J92', 'WNK4_HUMAN'))
        self.assertEqual(record.dr_false_neg[308], ('Q80UE6', 'WNK4_MOUSE'))
        self.assertEqual(record.dr_false_neg[309], ('Q7TPK6', 'WNK4_RAT'))
        self.assertEqual(record.dr_false_neg[310], ('Q9SZ67', 'WRK19_ARATH'))
        self.assertEqual(record.dr_false_neg[311], ('P83097', 'WSCK_DROME'))
        self.assertEqual(record.dr_false_neg[312], ('Q10447', 'YDEE_SCHPO'))
        self.assertEqual(record.dr_false_neg[313], ('Q5UQC1', 'YL232_MIMIV'))
        self.assertEqual(record.dr_false_neg[314], ('Q5UPU3', 'YL268_MIMIV'))
        self.assertEqual(record.dr_false_neg[315], ('P34516', 'YMX8_CAEEL'))
        self.assertEqual(record.dr_false_neg[316], ('Q09437', 'YP62_CAEEL'))
        self.assertEqual(record.dr_false_neg[317], ('Q9RI12', 'YPKA_YERPE'))
        self.assertEqual(record.dr_false_neg[318], ('Q05608', 'YPKA_YERPS'))
        self.assertEqual(record.dr_false_neg[319], ('Q09499', 'YQG4_CAEEL'))
        self.assertEqual(record.dr_false_neg[320], ('Q09298', 'YQO9_CAEEL'))
        self.assertEqual(record.dr_false_neg[321], ('Q7T6Y1', 'YR436_MIMIV'))
        self.assertEqual(record.dr_false_neg[322], ('Q09595', 'YRL5_CAEEL'))
        self.assertEqual(record.dr_false_neg[323], ('Q19238', 'YS3J_CAEEL'))

    def read1_false_pos(self, record):
        self.assertEqual(len(record.dr_false_pos), 83)
        self.assertEqual(record.dr_false_pos[0], ('Q7G9P4', 'ALDO3_ARATH'))
        self.assertEqual(record.dr_false_pos[1], ('Q7G191', 'ALDO4_ARATH'))
        self.assertEqual(record.dr_false_pos[2], ('Q7TUD1', 'ARGB_PROMP'))
        self.assertEqual(record.dr_false_pos[3], ('Q89FC2', 'ATKA_BRAJA'))
        self.assertEqual(record.dr_false_pos[4], ('Q62IJ6', 'ATKA_BURMA'))
        self.assertEqual(record.dr_false_pos[5], ('Q63VS3', 'ATKA_BURPS'))
        self.assertEqual(record.dr_false_pos[6], ('Q6N5H0', 'ATKA_RHOPA'))
        self.assertEqual(record.dr_false_pos[7], ('Q9ZB63', 'CARY_BACST'))
        self.assertEqual(record.dr_false_pos[8], ('P13351', 'CCNB2_XENLA'))
        self.assertEqual(record.dr_false_pos[9], ('P08920', 'CD2_MOUSE'))
        self.assertEqual(record.dr_false_pos[10], ('P39525', 'CEM1_YEAST'))
        self.assertEqual(record.dr_false_pos[11], ('Q9FKZ0', 'DRL43_ARATH'))
        self.assertEqual(record.dr_false_pos[12], ('Q8NG57', 'ELOA3_HUMAN'))
        self.assertEqual(record.dr_false_pos[13], ('P31789', 'ENV_IPMAE'))
        self.assertEqual(record.dr_false_pos[14], ('P56902', 'FABF_RHIME'))
        self.assertEqual(record.dr_false_pos[15], ('P40801', 'GAL10_PACTA'))
        self.assertEqual(record.dr_false_pos[16], ('Q43070', 'GALE1_PEA'))
        self.assertEqual(record.dr_false_pos[17], ('Q5N3K9', 'GLGC_SYNP6'))
        self.assertEqual(record.dr_false_pos[18], ('P27584', 'GPA1_SCHPO'))
        self.assertEqual(record.dr_false_pos[19], ('O33819', 'HCRA_THAAR'))
        self.assertEqual(record.dr_false_pos[20], ('Q8K9W9', 'HEMK_BUCAP'))
        self.assertEqual(record.dr_false_pos[21], ('O74713', 'HGT1_CANAL'))
        self.assertEqual(record.dr_false_pos[22], ('Q8TJA8', 'HPPA2_METAC'))
        self.assertEqual(record.dr_false_pos[23], ('Q8PYZ7', 'HPPA2_METMA'))
        self.assertEqual(record.dr_false_pos[24], ('P11679', 'K2C8_MOUSE'))
        self.assertEqual(record.dr_false_pos[25], ('Q8R711', 'KHSE_THETN'))
        self.assertEqual(record.dr_false_pos[26], ('P34038', 'KPYK_LACDE'))
        self.assertEqual(record.dr_false_pos[27], ('P06624', 'MIP_BOVIN'))
        self.assertEqual(record.dr_false_pos[28], ('Q6RZ07', 'MIP_CAVPO'))
        self.assertEqual(record.dr_false_pos[29], ('P30301', 'MIP_HUMAN'))
        self.assertEqual(record.dr_false_pos[30], ('P51180', 'MIP_MOUSE'))
        self.assertEqual(record.dr_false_pos[31], ('P09011', 'MIP_RAT'))
        self.assertEqual(record.dr_false_pos[32], ('Q6J8I9', 'MIP_SHEEP'))
        self.assertEqual(record.dr_false_pos[33], ('Q83HJ6', 'MRAW_TROW8'))
        self.assertEqual(record.dr_false_pos[34], ('Q83GN7', 'MRAW_TROWT'))
        self.assertEqual(record.dr_false_pos[35], ('P33527', 'MRP1_HUMAN'))
        self.assertEqual(record.dr_false_pos[36], ('P45060', 'MURE_HAEIN'))
        self.assertEqual(record.dr_false_pos[37], ('Q9UQQ1', 'NALDL_HUMAN'))
        self.assertEqual(record.dr_false_pos[38], ('Q8DGQ5', 'NANE_SYNEL'))
        self.assertEqual(record.dr_false_pos[39], ('O66841', 'NUOF_AQUAE'))
        self.assertEqual(record.dr_false_pos[40], ('O15018', 'PDZK3_HUMAN'))
        self.assertEqual(record.dr_false_pos[41], ('Q9QZR8', 'PDZK3_RAT'))
        self.assertEqual(record.dr_false_pos[42], ('Q863Y8', 'PEVR2_PAPHA'))
        self.assertEqual(record.dr_false_pos[43], ('P09607', 'PME21_LYCES'))
        self.assertEqual(record.dr_false_pos[44], ('Q96575', 'PME22_LYCES'))
        self.assertEqual(record.dr_false_pos[45], ('Q9RC08', 'PSAA_PROMP'))
        self.assertEqual(record.dr_false_pos[46], ('P56580', 'PTHB_ECOLI'))
        self.assertEqual(record.dr_false_pos[47], ('O32522', 'PTHB_ERWAM'))
        self.assertEqual(record.dr_false_pos[48], ('Q83H66', 'PURL_TROW8'))
        self.assertEqual(record.dr_false_pos[49], ('Q83FE8', 'PURL_TROWT'))
        self.assertEqual(record.dr_false_pos[50], ('P91341', 'PWP2_CAEEL'))
        self.assertEqual(record.dr_false_pos[51], ('Q13637', 'RAB32_HUMAN'))
        self.assertEqual(record.dr_false_pos[52], ('P57729', 'RAB38_HUMAN'))
        self.assertEqual(record.dr_false_pos[53], ('Q8QZZ8', 'RAB38_MOUSE'))
        self.assertEqual(record.dr_false_pos[54], ('Q4URF8', 'RL15_XANC8'))
        self.assertEqual(record.dr_false_pos[55], ('Q8PC34', 'RL15_XANCP'))
        self.assertEqual(record.dr_false_pos[56], ('Q601F3', 'RL27_MYCH2'))
        self.assertEqual(record.dr_false_pos[57], ('Q9V0L6', 'RPIA_PYRAB'))
        self.assertEqual(record.dr_false_pos[58], ('Q5LTT5', 'RPIA_SILPO'))
        self.assertEqual(record.dr_false_pos[59], ('P33540', 'RPOP_NEUCR'))
        self.assertEqual(record.dr_false_pos[60], ('P41638', 'RR4_PINTH'))
        self.assertEqual(record.dr_false_pos[61], ('Q7NRT4', 'RS9_CHRVO'))
        self.assertEqual(record.dr_false_pos[62], ('Q982W9', 'RS9_RHILO'))
        self.assertEqual(record.dr_false_pos[63], ('Q7M7R1', 'RS9_WOLSU'))
        self.assertEqual(record.dr_false_pos[64], ('Q4JAZ8', 'SPEE_SULAC'))
        self.assertEqual(record.dr_false_pos[65], ('Q9N0T1', 'STC1_BOVIN'))
        self.assertEqual(record.dr_false_pos[66], ('P52823', 'STC1_HUMAN'))
        self.assertEqual(record.dr_false_pos[67], ('O55183', 'STC1_MOUSE'))
        self.assertEqual(record.dr_false_pos[68], ('P97574', 'STC1_RAT'))
        self.assertEqual(record.dr_false_pos[69], ('P78371', 'TCPB_HUMAN'))
        self.assertEqual(record.dr_false_pos[70], ('P80314', 'TCPB_MOUSE'))
        self.assertEqual(record.dr_false_pos[71], ('P42943', 'TCPH_YEAST'))
        self.assertEqual(record.dr_false_pos[72], ('Q8VIM0', 'TIMD3_MOUSE'))
        self.assertEqual(record.dr_false_pos[73], ('Q9WVB2', 'TLE2_MOUSE'))
        self.assertEqual(record.dr_false_pos[74], ('Q02880', 'TOP2B_HUMAN'))
        self.assertEqual(record.dr_false_pos[75], ('Q9K715', 'TPIS_BACHD'))
        self.assertEqual(record.dr_false_pos[76], ('Q9GMA3', 'VSX1_BOVIN'))
        self.assertEqual(record.dr_false_pos[77], ('P29944', 'YCB2_PSEDE'))
        self.assertEqual(record.dr_false_pos[78], ('P33222', 'YJFC_ECOLI'))
        self.assertEqual(record.dr_false_pos[79], ('Q09371', 'YS42_CAEEL'))
        self.assertEqual(record.dr_false_pos[80], ('O32095', 'YUEF_BACSU'))
        self.assertEqual(record.dr_false_pos[81], ('P47917', 'ZRP4_MAIZE'))
        self.assertEqual(record.dr_false_pos[82], ('Q9PIN2', 'ZUPT_CAMJE'))

    def read1_potential(self, record):
        self.assertEqual(len(record.dr_potential), 30)
        self.assertEqual(record.dr_potential[0], ('Q02066', 'AAIP_WHEAT'))
        self.assertEqual(record.dr_potential[1], ('Q09136', 'AAPK1_PIG'))
        self.assertEqual(record.dr_potential[2], ('P11681', 'ABL_CALVI'))
        self.assertEqual(record.dr_potential[3], ('P19026', 'CDC21_PEA'))
        self.assertEqual(record.dr_potential[4], ('P28567', 'CDC22_PEA'))
        self.assertEqual(record.dr_potential[5], ('Q38773', 'CDC2B_ANTMA'))
        self.assertEqual(record.dr_potential[6], ('P43290', 'CDC2_PETHY'))
        self.assertEqual(record.dr_potential[7], ('P13387', 'EGFR_CHICK'))
        self.assertEqual(record.dr_potential[8], ('P55245', 'EGFR_MACMU'))
        self.assertEqual(record.dr_potential[9], ('Q61527', 'ERBB4_MOUSE'))
        self.assertEqual(record.dr_potential[10], ('Q29000', 'IGF1R_PIG'))
        self.assertEqual(record.dr_potential[11], ('Q64716', 'INSRR_RAT'))
        self.assertEqual(record.dr_potential[12], ('Q28516', 'INSR_MACMU'))
        self.assertEqual(record.dr_potential[13], ('P80197', 'KAFK_PHYPO'))
        self.assertEqual(record.dr_potential[14], ('O19175', 'KC1A_PIG'))
        self.assertEqual(record.dr_potential[15], ('P35508', 'KC1D_BOVIN'))
        self.assertEqual(record.dr_potential[16], ('P81123', 'KC1D_RABIT'))
        self.assertEqual(record.dr_potential[17], ('Q01621', 'LCK_RAT'))
        self.assertEqual(record.dr_potential[18], ('P49136', 'MAPK2_CRILO'))
        self.assertEqual(record.dr_potential[19], ('Q63454', 'MK04_RAT'))
        self.assertEqual(record.dr_potential[20], ('P39746', 'MP2K3_XENLA'))
        self.assertEqual(record.dr_potential[21], ('Q8VCR8', 'MYLK2_MOUSE'))
        self.assertEqual(record.dr_potential[22], ('P79280', 'MYLK_PIG'))
        self.assertEqual(record.dr_potential[23], ('O02827', 'MYLK_SHEEP'))
        self.assertEqual(record.dr_potential[24], ('P34101', 'PK1_DICDI'))
        self.assertEqual(record.dr_potential[25], ('P34102', 'PK3_DICDI'))
        self.assertEqual(record.dr_potential[26], ('P34103', 'PK4_DICDI'))
        self.assertEqual(record.dr_potential[27], ('P34104', 'PK5_DICDI'))
        self.assertEqual(record.dr_potential[28], ('O73792', 'TIE1_BRARE'))
        self.assertEqual(record.dr_potential[29], ('P34633', 'YOO1_CAEEL'))
        self.assertEqual(record.dr_unknown, [])
        self.assertEqual(len(record.pdb_structs), 331)
        self.assertEqual(record.pdb_structs[0], '1A9U')
        self.assertEqual(record.pdb_structs[1], '1AD5')
        self.assertEqual(record.pdb_structs[2], '1AGW')
        self.assertEqual(record.pdb_structs[3], '1APM')
        self.assertEqual(record.pdb_structs[4], '1ATP')
        self.assertEqual(record.pdb_structs[5], '1B6C')
        self.assertEqual(record.pdb_structs[6], '1BKX')
        self.assertEqual(record.pdb_structs[7], '1BL6')
        self.assertEqual(record.pdb_structs[8], '1BL7')
        self.assertEqual(record.pdb_structs[9], '1BLX')
        self.assertEqual(record.pdb_structs[10], '1BMK')
        self.assertEqual(record.pdb_structs[11], '1BX6')
        self.assertEqual(record.pdb_structs[12], '1BYG')
        self.assertEqual(record.pdb_structs[13], '1CKI')
        self.assertEqual(record.pdb_structs[14], '1CKJ')
        self.assertEqual(record.pdb_structs[15], '1CSN')
        self.assertEqual(record.pdb_structs[16], '1CTP')
        self.assertEqual(record.pdb_structs[17], '1DAW')
        self.assertEqual(record.pdb_structs[18], '1DAY')
        self.assertEqual(record.pdb_structs[19], '1DI9')
        self.assertEqual(record.pdb_structs[20], '1DS5')
        self.assertEqual(record.pdb_structs[21], '1E9H')
        self.assertEqual(record.pdb_structs[22], '1EH4')
        self.assertEqual(record.pdb_structs[23], '1ERK')
        self.assertEqual(record.pdb_structs[24], '1F0Q')
        self.assertEqual(record.pdb_structs[25], '1FGI')
        self.assertEqual(record.pdb_structs[26], '1FGK')
        self.assertEqual(record.pdb_structs[27], '1FIN')
        self.assertEqual(record.pdb_structs[28], '1FMK')
        self.assertEqual(record.pdb_structs[29], '1FMO')
        self.assertEqual(record.pdb_structs[30], '1FOT')
        self.assertEqual(record.pdb_structs[31], '1FPU')
        self.assertEqual(record.pdb_structs[32], '1FQ1')
        self.assertEqual(record.pdb_structs[33], '1FVV')
        self.assertEqual(record.pdb_structs[34], '1G3N')
        self.assertEqual(record.pdb_structs[35], '1GAG')
        self.assertEqual(record.pdb_structs[36], '1GJO')
        self.assertEqual(record.pdb_structs[37], '1GNG')
        self.assertEqual(record.pdb_structs[38], '1GOL')
        self.assertEqual(record.pdb_structs[39], '1GY3')
        self.assertEqual(record.pdb_structs[40], '1H1P')
        self.assertEqual(record.pdb_structs[41], '1H1Q')
        self.assertEqual(record.pdb_structs[42], '1H1R')
        self.assertEqual(record.pdb_structs[43], '1H1S')
        self.assertEqual(record.pdb_structs[44], '1H1W')
        self.assertEqual(record.pdb_structs[45], '1H24')
        self.assertEqual(record.pdb_structs[46], '1H25')
        self.assertEqual(record.pdb_structs[47], '1H26')
        self.assertEqual(record.pdb_structs[48], '1H27')
        self.assertEqual(record.pdb_structs[49], '1H28')
        self.assertEqual(record.pdb_structs[50], '1H8F')
        self.assertEqual(record.pdb_structs[51], '1HOW')
        self.assertEqual(record.pdb_structs[52], '1I09')
        self.assertEqual(record.pdb_structs[53], '1I44')
        self.assertEqual(record.pdb_structs[54], '1IAN')
        self.assertEqual(record.pdb_structs[55], '1IAS')
        self.assertEqual(record.pdb_structs[56], '1IEP')
        self.assertEqual(record.pdb_structs[57], '1IG1')
        self.assertEqual(record.pdb_structs[58], '1IR3')
        self.assertEqual(record.pdb_structs[59], '1IRK')
        self.assertEqual(record.pdb_structs[60], '1J1B')
        self.assertEqual(record.pdb_structs[61], '1J1C')
        self.assertEqual(record.pdb_structs[62], '1J3H')
        self.assertEqual(record.pdb_structs[63], '1J91')
        self.assertEqual(record.pdb_structs[64], '1JAM')
        self.assertEqual(record.pdb_structs[65], '1JKK')
        self.assertEqual(record.pdb_structs[66], '1JKL')
        self.assertEqual(record.pdb_structs[67], '1JKS')
        self.assertEqual(record.pdb_structs[68], '1JKT')
        self.assertEqual(record.pdb_structs[69], '1JLU')
        self.assertEqual(record.pdb_structs[70], '1JPA')
        self.assertEqual(record.pdb_structs[71], '1JQH')
        self.assertEqual(record.pdb_structs[72], '1JST')
        self.assertEqual(record.pdb_structs[73], '1JWH')
        self.assertEqual(record.pdb_structs[74], '1K2P')
        self.assertEqual(record.pdb_structs[75], '1K3A')
        self.assertEqual(record.pdb_structs[76], '1K9A')
        self.assertEqual(record.pdb_structs[77], '1KMU')
        self.assertEqual(record.pdb_structs[78], '1KMW')
        self.assertEqual(record.pdb_structs[79], '1KSW')
        self.assertEqual(record.pdb_structs[80], '1KV1')
        self.assertEqual(record.pdb_structs[81], '1KV2')
        self.assertEqual(record.pdb_structs[82], '1KWP')
        self.assertEqual(record.pdb_structs[83], '1L3R')
        self.assertEqual(record.pdb_structs[84], '1LC9')
        self.assertEqual(record.pdb_structs[85], '1LCH')
        self.assertEqual(record.pdb_structs[86], '1LD2')
        self.assertEqual(record.pdb_structs[87], '1LEW')
        self.assertEqual(record.pdb_structs[88], '1LEZ')
        self.assertEqual(record.pdb_structs[89], '1LFN')
        self.assertEqual(record.pdb_structs[90], '1LFR')
        self.assertEqual(record.pdb_structs[91], '1LG3')
        self.assertEqual(record.pdb_structs[92], '1LHX')
        self.assertEqual(record.pdb_structs[93], '1LP4')
        self.assertEqual(record.pdb_structs[94], '1LPU')
        self.assertEqual(record.pdb_structs[95], '1LR4')
        self.assertEqual(record.pdb_structs[96], '1LUF')
        self.assertEqual(record.pdb_structs[97], '1LWP')
        self.assertEqual(record.pdb_structs[98], '1M14')
        self.assertEqual(record.pdb_structs[99], '1M17')
        self.assertEqual(record.pdb_structs[100], '1M2P')
        self.assertEqual(record.pdb_structs[101], '1M2Q')
        self.assertEqual(record.pdb_structs[102], '1M2R')
        self.assertEqual(record.pdb_structs[103], '1M52')
        self.assertEqual(record.pdb_structs[104], '1M7N')
        self.assertEqual(record.pdb_structs[105], '1M7Q')
        self.assertEqual(record.pdb_structs[106], '1MP8')
        self.assertEqual(record.pdb_structs[107], '1MQ4')
        self.assertEqual(record.pdb_structs[108], '1MQB')
        self.assertEqual(record.pdb_structs[109], '1MRU')
        self.assertEqual(record.pdb_structs[110], '1MUO')
        self.assertEqual(record.pdb_structs[111], '1NA7')
        self.assertEqual(record.pdb_structs[112], '1NXK')
        self.assertEqual(record.pdb_structs[113], '1NY3')
        self.assertEqual(record.pdb_structs[114], '1O6K')
        self.assertEqual(record.pdb_structs[115], '1O6L')
        self.assertEqual(record.pdb_structs[116], '1O6Y')
        self.assertEqual(record.pdb_structs[117], '1O9U')
        self.assertEqual(record.pdb_structs[118], '1OB3')
        self.assertEqual(record.pdb_structs[119], '1OEC')
        self.assertEqual(record.pdb_structs[120], '1OGU')
        self.assertEqual(record.pdb_structs[121], '1OI9')
        self.assertEqual(record.pdb_structs[122], '1OIU')
        self.assertEqual(record.pdb_structs[123], '1OIY')
        self.assertEqual(record.pdb_structs[124], '1OKU')
        self.assertEqual(record.pdb_structs[125], '1OKV')
        self.assertEqual(record.pdb_structs[126], '1OKW')
        self.assertEqual(record.pdb_structs[127], '1OKY')
        self.assertEqual(record.pdb_structs[128], '1OKZ')
        self.assertEqual(record.pdb_structs[129], '1OL1')
        self.assertEqual(record.pdb_structs[130], '1OL2')
        self.assertEqual(record.pdb_structs[131], '1OL5')
        self.assertEqual(record.pdb_structs[132], '1OL6')
        self.assertEqual(record.pdb_structs[133], '1OL7')
        self.assertEqual(record.pdb_structs[134], '1OM1')
        self.assertEqual(record.pdb_structs[135], '1OMW')
        self.assertEqual(record.pdb_structs[136], '1OPJ')
        self.assertEqual(record.pdb_structs[137], '1OPK')
        self.assertEqual(record.pdb_structs[138], '1OPL')
        self.assertEqual(record.pdb_structs[139], '1OUK')
        self.assertEqual(record.pdb_structs[140], '1OUY')
        self.assertEqual(record.pdb_structs[141], '1OVE')
        self.assertEqual(record.pdb_structs[142], '1OZ1')
        self.assertEqual(record.pdb_structs[143], '1P14')
        self.assertEqual(record.pdb_structs[144], '1P38')
        self.assertEqual(record.pdb_structs[145], '1P4F')
        self.assertEqual(record.pdb_structs[146], '1P4O')
        self.assertEqual(record.pdb_structs[147], '1P5E')
        self.assertEqual(record.pdb_structs[148], '1PF6')
        self.assertEqual(record.pdb_structs[149], '1PF8')
        self.assertEqual(record.pdb_structs[150], '1PHK')
        self.assertEqual(record.pdb_structs[151], '1PJK')
        self.assertEqual(record.pdb_structs[152], '1PKD')
        self.assertEqual(record.pdb_structs[153], '1PKG')
        self.assertEqual(record.pdb_structs[154], '1PME')
        self.assertEqual(record.pdb_structs[155], '1PVK')
        self.assertEqual(record.pdb_structs[156], '1PY5')
        self.assertEqual(record.pdb_structs[157], '1PYX')
        self.assertEqual(record.pdb_structs[158], '1Q24')
        self.assertEqual(record.pdb_structs[159], '1Q3D')
        self.assertEqual(record.pdb_structs[160], '1Q3W')
        self.assertEqual(record.pdb_structs[161], '1Q41')
        self.assertEqual(record.pdb_structs[162], '1Q4L')
        self.assertEqual(record.pdb_structs[163], '1Q5K')
        self.assertEqual(record.pdb_structs[164], '1Q61')
        self.assertEqual(record.pdb_structs[165], '1Q62')
        self.assertEqual(record.pdb_structs[166], '1Q8T')
        self.assertEqual(record.pdb_structs[167], '1Q8U')
        self.assertEqual(record.pdb_structs[168], '1Q8W')
        self.assertEqual(record.pdb_structs[169], '1Q8Y')
        self.assertEqual(record.pdb_structs[170], '1Q8Z')
        self.assertEqual(record.pdb_structs[171], '1Q97')
        self.assertEqual(record.pdb_structs[172], '1Q99')
        self.assertEqual(record.pdb_structs[173], '1QCF')
        self.assertEqual(record.pdb_structs[174], '1QL6')
        self.assertEqual(record.pdb_structs[175], '1QMZ')
        self.assertEqual(record.pdb_structs[176], '1QPC')
        self.assertEqual(record.pdb_structs[177], '1QPE')
        self.assertEqual(record.pdb_structs[178], '1QPJ')
        self.assertEqual(record.pdb_structs[179], '1R0E')
        self.assertEqual(record.pdb_structs[180], '1R0P')
        self.assertEqual(record.pdb_structs[181], '1R1W')
        self.assertEqual(record.pdb_structs[182], '1R39')
        self.assertEqual(record.pdb_structs[183], '1R3C')
        self.assertEqual(record.pdb_structs[184], '1RDQ')
        self.assertEqual(record.pdb_structs[185], '1RE8')
        self.assertEqual(record.pdb_structs[186], '1REJ')
        self.assertEqual(record.pdb_structs[187], '1REK')
        self.assertEqual(record.pdb_structs[188], '1RJB')
        self.assertEqual(record.pdb_structs[189], '1RQQ')
        self.assertEqual(record.pdb_structs[190], '1RW8')
        self.assertEqual(record.pdb_structs[191], '1S9I')
        self.assertEqual(record.pdb_structs[192], '1S9J')
        self.assertEqual(record.pdb_structs[193], '1SM2')
        self.assertEqual(record.pdb_structs[194], '1SMH')
        self.assertEqual(record.pdb_structs[195], '1SNU')
        self.assertEqual(record.pdb_structs[196], '1SNX')
        self.assertEqual(record.pdb_structs[197], '1STC')
        self.assertEqual(record.pdb_structs[198], '1SZM')
        self.assertEqual(record.pdb_structs[199], '1T45')
        self.assertEqual(record.pdb_structs[200], '1T46')
        self.assertEqual(record.pdb_structs[201], '1TVO')
        self.assertEqual(record.pdb_structs[202], '1U59')
        self.assertEqual(record.pdb_structs[203], '1U5Q')
        self.assertEqual(record.pdb_structs[204], '1U5R')
        self.assertEqual(record.pdb_structs[205], '1U7E')
        self.assertEqual(record.pdb_structs[206], '1UNL')
        self.assertEqual(record.pdb_structs[207], '1URC')
        self.assertEqual(record.pdb_structs[208], '1UU3')
        self.assertEqual(record.pdb_structs[209], '1UU7')
        self.assertEqual(record.pdb_structs[210], '1UU8')
        self.assertEqual(record.pdb_structs[211], '1UU9')
        self.assertEqual(record.pdb_structs[212], '1UV5')
        self.assertEqual(record.pdb_structs[213], '1UVR')
        self.assertEqual(record.pdb_structs[214], '1UWH')
        self.assertEqual(record.pdb_structs[215], '1UWJ')
        self.assertEqual(record.pdb_structs[216], '1V0B')
        self.assertEqual(record.pdb_structs[217], '1V0P')
        self.assertEqual(record.pdb_structs[218], '1VJY')
        self.assertEqual(record.pdb_structs[219], '1VR2')
        self.assertEqual(record.pdb_structs[220], '1VYW')
        self.assertEqual(record.pdb_structs[221], '1VZO')
        self.assertEqual(record.pdb_structs[222], '1W7H')
        self.assertEqual(record.pdb_structs[223], '1W82')
        self.assertEqual(record.pdb_structs[224], '1W83')
        self.assertEqual(record.pdb_structs[225], '1W84')
        self.assertEqual(record.pdb_structs[226], '1W98')
        self.assertEqual(record.pdb_structs[227], '1WBN')
        self.assertEqual(record.pdb_structs[228], '1WBO')
        self.assertEqual(record.pdb_structs[229], '1WBP')
        self.assertEqual(record.pdb_structs[230], '1WBS')
        self.assertEqual(record.pdb_structs[231], '1WBT')
        self.assertEqual(record.pdb_structs[232], '1WBV')
        self.assertEqual(record.pdb_structs[233], '1WBW')
        self.assertEqual(record.pdb_structs[234], '1WFC')
        self.assertEqual(record.pdb_structs[235], '1WMK')
        self.assertEqual(record.pdb_structs[236], '1WZY')
        self.assertEqual(record.pdb_structs[237], '1X8B')
        self.assertEqual(record.pdb_structs[238], '1XH4')
        self.assertEqual(record.pdb_structs[239], '1XH5')
        self.assertEqual(record.pdb_structs[240], '1XH6')
        self.assertEqual(record.pdb_structs[241], '1XH7')
        self.assertEqual(record.pdb_structs[242], '1XH8')
        self.assertEqual(record.pdb_structs[243], '1XH9')
        self.assertEqual(record.pdb_structs[244], '1XHA')
        self.assertEqual(record.pdb_structs[245], '1XJD')
        self.assertEqual(record.pdb_structs[246], '1XKK')
        self.assertEqual(record.pdb_structs[247], '1XO2')
        self.assertEqual(record.pdb_structs[248], '1XQZ')
        self.assertEqual(record.pdb_structs[249], '1XR1')
        self.assertEqual(record.pdb_structs[250], '1XWS')
        self.assertEqual(record.pdb_structs[251], '1Y57')
        self.assertEqual(record.pdb_structs[252], '1Y6B')
        self.assertEqual(record.pdb_structs[253], '1YDR')
        self.assertEqual(record.pdb_structs[254], '1YDS')
        self.assertEqual(record.pdb_structs[255], '1YDT')
        self.assertEqual(record.pdb_structs[256], '1YHS')
        self.assertEqual(record.pdb_structs[257], '1YI3')
        self.assertEqual(record.pdb_structs[258], '1YI4')
        self.assertEqual(record.pdb_structs[259], '1YKR')
        self.assertEqual(record.pdb_structs[260], '1YM7')
        self.assertEqual(record.pdb_structs[261], '1YMI')
        self.assertEqual(record.pdb_structs[262], '1YOL')
        self.assertEqual(record.pdb_structs[263], '1YOM')
        self.assertEqual(record.pdb_structs[264], '1YVJ')
        self.assertEqual(record.pdb_structs[265], '1YW2')
        self.assertEqual(record.pdb_structs[266], '1YWR')
        self.assertEqual(record.pdb_structs[267], '1Z57')
        self.assertEqual(record.pdb_structs[268], '1Z5M')
        self.assertEqual(record.pdb_structs[269], '1ZOE')
        self.assertEqual(record.pdb_structs[270], '1ZOG')
        self.assertEqual(record.pdb_structs[271], '1ZOH')
        self.assertEqual(record.pdb_structs[272], '1ZRZ')
        self.assertEqual(record.pdb_structs[273], '1ZXE')
        self.assertEqual(record.pdb_structs[274], '1ZY4')
        self.assertEqual(record.pdb_structs[275], '1ZY5')
        self.assertEqual(record.pdb_structs[276], '1ZYC')
        self.assertEqual(record.pdb_structs[277], '1ZYD')
        self.assertEqual(record.pdb_structs[278], '1ZZ2')
        self.assertEqual(record.pdb_structs[279], '1ZZL')
        self.assertEqual(record.pdb_structs[280], '2A19')
        self.assertEqual(record.pdb_structs[281], '2A1A')
        self.assertEqual(record.pdb_structs[282], '2AC3')
        self.assertEqual(record.pdb_structs[283], '2AC5')
        self.assertEqual(record.pdb_structs[284], '2AUH')
        self.assertEqual(record.pdb_structs[285], '2B4S')
        self.assertEqual(record.pdb_structs[286], '2B54')
        self.assertEqual(record.pdb_structs[287], '2B7A')
        self.assertEqual(record.pdb_structs[288], '2B9F')
        self.assertEqual(record.pdb_structs[289], '2B9H')
        self.assertEqual(record.pdb_structs[290], '2B9I')
        self.assertEqual(record.pdb_structs[291], '2B9J')
        self.assertEqual(record.pdb_structs[292], '2BAJ')
        self.assertEqual(record.pdb_structs[293], '2BAK')
        self.assertEqual(record.pdb_structs[294], '2BAL')
        self.assertEqual(record.pdb_structs[295], '2BAQ')
        self.assertEqual(record.pdb_structs[296], '2BCJ')
        self.assertEqual(record.pdb_structs[297], '2BIK')
        self.assertEqual(record.pdb_structs[298], '2BIL')
        self.assertEqual(record.pdb_structs[299], '2BIY')
        self.assertEqual(record.pdb_structs[300], '2BMC')
        self.assertEqual(record.pdb_structs[301], '2BPM')
        self.assertEqual(record.pdb_structs[302], '2BZH')
        self.assertEqual(record.pdb_structs[303], '2BZI')
        self.assertEqual(record.pdb_structs[304], '2BZJ')
        self.assertEqual(record.pdb_structs[305], '2C1A')
        self.assertEqual(record.pdb_structs[306], '2C1B')
        self.assertEqual(record.pdb_structs[307], '2C3I')
        self.assertEqual(record.pdb_structs[308], '2C4G')
        self.assertEqual(record.pdb_structs[309], '2C6D')
        self.assertEqual(record.pdb_structs[310], '2C6E')
        self.assertEqual(record.pdb_structs[311], '2C6T')
        self.assertEqual(record.pdb_structs[312], '2CPK')
        self.assertEqual(record.pdb_structs[313], '2CSN')
        self.assertEqual(record.pdb_structs[314], '2ERK')
        self.assertEqual(record.pdb_structs[315], '2ERZ')
        self.assertEqual(record.pdb_structs[316], '2ESM')
        self.assertEqual(record.pdb_structs[317], '2ETK')
        self.assertEqual(record.pdb_structs[318], '2ETO')
        self.assertEqual(record.pdb_structs[319], '2ETR')
        self.assertEqual(record.pdb_structs[320], '2EU9')
        self.assertEqual(record.pdb_structs[321], '2EXM')
        self.assertEqual(record.pdb_structs[322], '2F4J')
        self.assertEqual(record.pdb_structs[323], '2F57')
        self.assertEqual(record.pdb_structs[324], '2FGI')
        self.assertEqual(record.pdb_structs[325], '2HCK')
        self.assertEqual(record.pdb_structs[326], '2PHK')
        self.assertEqual(record.pdb_structs[327], '2SRC')
        self.assertEqual(record.pdb_structs[328], '3ERK')
        self.assertEqual(record.pdb_structs[329], '3LCK')
        self.assertEqual(record.pdb_structs[330], '4ERK')

    def test_read2(self):
        "Parsing Prosite record ps00159.txt"
        filename = os.path.join('Prosite', 'ps00159.txt')
        handle = open(filename)
        record = Prosite.read(handle)
        handle.close()
        self.assertEqual(record.name, "ALDOLASE_KDPG_KHG_1")
        self.assertEqual(record.type, "PATTERN")
        self.assertEqual(record.accession, "PS00159")
        self.assertEqual(record.created, "APR-1990")
        self.assertEqual(record.data_update, "NOV-1995")
        self.assertEqual(record.info_update, "JUL-2005")
        self.assertEqual(record.pdoc, "PDOC00144")
        self.assertEqual(record.description, "KDPG and KHG aldolases active site.")
        self.assertEqual(record.pattern, "G-[LIVM]-x(3)-E-[LIV]-T-[LF]-R.")
        self.assertEqual(record.matrix, [])
        self.assertEqual(record.rules, [])
        self.assertEqual(record.nr_sp_release, "49.3")
        self.assertEqual(record.nr_sp_seqs, 212425)
        self.assertEqual(record.cc_taxo_range, "???P?")
        self.assertEqual(record.cc_max_repeat, "1")
        self.assertEqual(len(record.cc_site), 1)
        self.assertEqual(record.cc_site[0], (8, 'active_site'))
        self.assertEqual(len(record.dr_positive), 9)
        self.assertEqual(record.dr_positive[0], ('P00885', 'ALKD_PSEPU'))
        self.assertEqual(record.dr_positive[1], ('P0A957', 'ALKH_ECO57'))
        self.assertEqual(record.dr_positive[2], ('P0A956', 'ALKH_ECOL6'))
        self.assertEqual(record.dr_positive[3], ('P0A955', 'ALKH_ECOLI'))
        self.assertEqual(record.dr_positive[4], ('P38448', 'ALKH_ERWCH'))
        self.assertEqual(record.dr_positive[5], ('P44480', 'ALKH_HAEIN'))
        self.assertEqual(record.dr_positive[6], ('P0A958', 'ALKH_SHIFL'))
        self.assertEqual(record.dr_positive[7], ('O83578', 'ALKH_TREPA'))
        self.assertEqual(record.dr_positive[8], ('Q00384', 'ALKH_ZYMMO'))
        self.assertEqual(len(record.dr_false_neg), 1)
        self.assertEqual(record.dr_false_neg[0], ('P50846', 'ALKH_BACSU'))
        self.assertEqual(len(record.dr_false_pos), 3)
        self.assertEqual(record.dr_false_pos[0], ('Q9WV82', 'TLR4_CRIGR'))
        self.assertEqual(record.dr_false_pos[1], ('Q9QUK6', 'TLR4_MOUSE'))
        self.assertEqual(record.dr_false_pos[2], ('Q9QX05', 'TLR4_RAT'))
        self.assertEqual(len(record.dr_potential), 0)
        self.assertEqual(len(record.dr_unknown), 0)
        self.assertEqual(len(record.pdb_structs), 6)
        self.assertEqual(record.pdb_structs[0], "1EUA")
        self.assertEqual(record.pdb_structs[1], "1EUN")
        self.assertEqual(record.pdb_structs[2], "1FQ0")
        self.assertEqual(record.pdb_structs[3], "1FWR")
        self.assertEqual(record.pdb_structs[4], "1MXS")
        self.assertEqual(record.pdb_structs[5], "1VHC")

    def test_read3(self):
        "Parsing Prosite record ps00165.txt"
        filename = os.path.join('Prosite', 'ps00165.txt')
        handle = open(filename)
        record = Prosite.read(handle)
        handle.close()
        self.assertEqual(record.name, "DEHYDRATASE_SER_THR")
        self.assertEqual(record.type, "PATTERN")
        self.assertEqual(record.accession, "PS00165")
        self.assertEqual(record.created, "APR-1990")
        self.assertEqual(record.data_update, "DEC-2004")
        self.assertEqual(record.info_update, "MAR-2006")
        self.assertEqual(record.pdoc, "PDOC00149")

        self.assertEqual(record.description, "Serine/threonine dehydratases pyridoxal-phosphate attachment site.")
        self.assertEqual(record.pattern, "[DESH]-x(4,5)-[STVG]-{EVKD}-[AS]-[FYI]-K-[DLIFSA]-[RLVMF]-[GA]-[LIVMGA].")
        self.assertEqual(record.matrix, [])
        self.assertEqual(record.rules, [])
        self.assertEqual(record.nr_sp_release, "49.3")
        self.assertEqual(record.nr_sp_seqs, 212425)
        self.assertEqual(record.cc_taxo_range, "A?EP?")
        self.assertEqual(record.cc_max_repeat, "1")
        self.assertEqual(len(record.cc_site), 1)
        self.assertEqual(record.cc_site[0], (7, 'pyridoxal_phosphate'))
        self.assertEqual(len(record.dr_positive), 77)
        self.assertEqual(record.dr_positive[0], ('Q6FDC1', 'SDHD_ACIAD'))
        self.assertEqual(record.dr_positive[1], ('Q81S85', 'SDHD_BACAN'))
        self.assertEqual(record.dr_positive[2], ('Q73AC5', 'SDHD_BACC1'))
        self.assertEqual(record.dr_positive[3], ('Q81F73', 'SDHD_BACCR'))
        self.assertEqual(record.dr_positive[4], ('Q63D23', 'SDHD_BACCZ'))
        self.assertEqual(record.dr_positive[5], ('Q9KC12', 'SDHD_BACHD'))
        self.assertEqual(record.dr_positive[6], ('Q6HKG3', 'SDHD_BACHK'))
        self.assertEqual(record.dr_positive[7], ('Q5WGL6', 'SDHD_BACSK'))
        self.assertEqual(record.dr_positive[8], ('Q75TH5', 'SDHD_BACST'))
        self.assertEqual(record.dr_positive[9], ('P54555', 'SDHD_BACSU'))
        self.assertEqual(record.dr_positive[10], ('Q7NUP9', 'SDHD_CHRVO'))
        self.assertEqual(record.dr_positive[11], ('Q8XCK5', 'SDHD_ECO57'))
        self.assertEqual(record.dr_positive[12], ('Q8FFF4', 'SDHD_ECOL6'))
        self.assertEqual(record.dr_positive[13], ('P00926', 'SDHD_ECOLI'))
        self.assertEqual(record.dr_positive[14], ('Q88TY1', 'SDHD_LACPL'))
        self.assertEqual(record.dr_positive[15], ('Q7N5H9', 'SDHD_PHOLL'))
        self.assertEqual(record.dr_positive[16], ('Q6LHF5', 'SDHD_PHOPR'))
        self.assertEqual(record.dr_positive[17], ('Q9HYN9', 'SDHD_PSEAE'))
        self.assertEqual(record.dr_positive[18], ('Q8Z2L4', 'SDHD_SALTI'))
        self.assertEqual(record.dr_positive[19], ('Q8ZL08', 'SDHD_SALTY'))
        self.assertEqual(record.dr_positive[20], ('Q83MK8', 'SDHD_SHIFL'))
        self.assertEqual(record.dr_positive[21], ('Q9KL72', 'SDHD_VIBCH'))
        self.assertEqual(record.dr_positive[22], ('Q87QA2', 'SDHD_VIBPA'))
        self.assertEqual(record.dr_positive[23], ('P20132', 'SDHL_HUMAN'))
        self.assertEqual(record.dr_positive[24], ('Q8VBT2', 'SDHL_MOUSE'))
        self.assertEqual(record.dr_positive[25], ('P09367', 'SDHL_RAT'))
        self.assertEqual(record.dr_positive[26], ('P17324', 'SDHL_YEAST'))
        self.assertEqual(record.dr_positive[27], ('Q9GZT4', 'SRR_HUMAN'))
        self.assertEqual(record.dr_positive[28], ('Q9QZX7', 'SRR_MOUSE'))
        self.assertEqual(record.dr_positive[29], ('P25379', 'STDH_YEAST'))
        self.assertEqual(record.dr_positive[30], ('Q9ZSS6', 'THD1_ARATH'))
        self.assertEqual(record.dr_positive[31], ('Q9KC63', 'THD1_BACHD'))
        self.assertEqual(record.dr_positive[32], ('P37946', 'THD1_BACSU'))
        self.assertEqual(record.dr_positive[33], ('P53607', 'THD1_BURCE'))
        self.assertEqual(record.dr_positive[34], ('Q04513', 'THD1_CORGL'))
        self.assertEqual(record.dr_positive[35], ('P04968', 'THD1_ECOLI'))
        self.assertEqual(record.dr_positive[36], ('P46493', 'THD1_HAEIN'))
        self.assertEqual(record.dr_positive[37], ('Q02145', 'THD1_LACLA'))
        self.assertEqual(record.dr_positive[38], ('P25306', 'THD1_LYCES'))
        self.assertEqual(record.dr_positive[39], ('P66898', 'THD1_MYCBO'))
        self.assertEqual(record.dr_positive[40], ('Q9X7F1', 'THD1_MYCLE'))
        self.assertEqual(record.dr_positive[41], ('P66897', 'THD1_MYCTU'))
        self.assertEqual(record.dr_positive[42], ('Q9CKJ2', 'THD1_PASMU'))
        self.assertEqual(record.dr_positive[43], ('P20506', 'THD1_SALTY'))
        self.assertEqual(record.dr_positive[44], ('P0AGF8', 'THD2_ECO57'))
        self.assertEqual(record.dr_positive[45], ('P0AGF7', 'THD2_ECOL6'))
        self.assertEqual(record.dr_positive[46], ('P0AGF6', 'THD2_ECOLI'))
        self.assertEqual(record.dr_positive[47], ('P11954', 'THD2_SALTY'))
        self.assertEqual(record.dr_positive[48], ('P0AGF9', 'THD2_SHIFL'))
        self.assertEqual(record.dr_positive[49], ('O42615', 'THDH_ARXAD'))
        self.assertEqual(record.dr_positive[50], ('P00927', 'THDH_YEAST'))
        self.assertEqual(record.dr_positive[51], ('Q9S7B5', 'THRC_ARATH'))
        self.assertEqual(record.dr_positive[52], ('Q00063', 'THRC_ASHGO'))
        self.assertEqual(record.dr_positive[53], ('Q9K7E3', 'THRC_BACHD'))
        self.assertEqual(record.dr_positive[54], ('P09123', 'THRC_BACSL'))
        self.assertEqual(record.dr_positive[55], ('P04990', 'THRC_BACSU'))
        self.assertEqual(record.dr_positive[56], ('P57289', 'THRC_BUCAI'))
        self.assertEqual(record.dr_positive[57], ('Q8K9V1', 'THRC_BUCAP'))
        self.assertEqual(record.dr_positive[58], ('Q89AR5', 'THRC_BUCBP'))
        self.assertEqual(record.dr_positive[59], ('P23669', 'THRC_CORGL'))
        self.assertEqual(record.dr_positive[60], ('P00934', 'THRC_ECOLI'))
        self.assertEqual(record.dr_positive[61], ('P44503', 'THRC_HAEIN'))
        self.assertEqual(record.dr_positive[62], ('Q9ZMX5', 'THRC_HELPJ'))
        self.assertEqual(record.dr_positive[63], ('O24924', 'THRC_HELPY'))
        self.assertEqual(record.dr_positive[64], ('P37145', 'THRC_METGL'))
        self.assertEqual(record.dr_positive[65], ('Q58860', 'THRC_METJA'))
        self.assertEqual(record.dr_positive[66], ('P66903', 'THRC_MYCBO'))
        self.assertEqual(record.dr_positive[67], ('P45837', 'THRC_MYCLE'))
        self.assertEqual(record.dr_positive[68], ('P66902', 'THRC_MYCTU'))
        self.assertEqual(record.dr_positive[69], ('P29363', 'THRC_PSEAE'))
        self.assertEqual(record.dr_positive[70], ('Q42598', 'THRC_SCHPO'))
        self.assertEqual(record.dr_positive[71], ('P27735', 'THRC_SERMA'))
        self.assertEqual(record.dr_positive[72], ('Q9MT28', 'THRC_SOLTU'))
        self.assertEqual(record.dr_positive[73], ('P74193', 'THRC_SYNY3'))
        self.assertEqual(record.dr_positive[74], ('P16120', 'THRC_YEAST'))
        self.assertEqual(record.dr_positive[75], ('P55664', 'Y4TJ_RHISN'))
        self.assertEqual(record.dr_positive[76], ('P36007', 'YKV8_YEAST'))
        self.assertEqual(len(record.dr_false_neg), 4)
        self.assertEqual(record.dr_false_neg[0], ('Q63IF8', 'SDHD_BURPS'))
        self.assertEqual(record.dr_false_neg[1], ('Q8RFX6', 'SDHD_FUSNN'))
        self.assertEqual(record.dr_false_neg[2], ('Q39469', 'THD1_CICAR'))
        self.assertEqual(record.dr_false_neg[3], ('O59791', 'YCNE_SCHPO'))
        self.assertEqual(len(record.dr_false_pos), 20)
        self.assertEqual(record.dr_false_pos[0], ('Q08043', 'ACTN3_HUMAN'))
        self.assertEqual(record.dr_false_pos[1], ('O88990', 'ACTN3_MOUSE'))
        self.assertEqual(record.dr_false_pos[2], ('P00885', 'ALKD_PSEPU'))
        self.assertEqual(record.dr_false_pos[3], ('Q8ZW89', 'AROE_PYRAE'))
        self.assertEqual(record.dr_false_pos[4], ('Q9S142', 'BFPB_ECO11'))
        self.assertEqual(record.dr_false_pos[5], ('Q47068', 'BFPB_ECO27'))
        self.assertEqual(record.dr_false_pos[6], ('P80549', 'CY551_CHRVI'))
        self.assertEqual(record.dr_false_pos[7], ('P00099', 'CY551_PSEAE'))
        self.assertEqual(record.dr_false_pos[8], ('P71128', 'CYSM_CAMJE'))
        self.assertEqual(record.dr_false_pos[9], ('Q9ZL20', 'DHOM_HELPJ'))
        self.assertEqual(record.dr_false_pos[10], ('P56429', 'DHOM_HELPY'))
        self.assertEqual(record.dr_false_pos[11], ('Q8BWT5', 'DIP2A_MOUSE'))
        self.assertEqual(record.dr_false_pos[12], ('Q72SI0', 'FEOB_LEPIC'))
        self.assertEqual(record.dr_false_pos[13], ('Q8F332', 'FEOB_LEPIN'))
        self.assertEqual(record.dr_false_pos[14], ('Q88AS0', 'GCST_PSESM'))
        self.assertEqual(record.dr_false_pos[15], ('Q8WVV9', 'HNRLL_HUMAN'))
        self.assertEqual(record.dr_false_pos[16], ('Q921F4', 'HNRLL_MOUSE'))
        self.assertEqual(record.dr_false_pos[17], ('P16540', 'KAS1_STRVN'))
        self.assertEqual(record.dr_false_pos[18], ('Q8UE36', 'RL15_AGRT5'))
        self.assertEqual(record.dr_false_pos[19], ('Q46XK7', 'RS20_RALEJ'))
        self.assertEqual(len(record.dr_potential), 1)
        self.assertEqual(record.dr_potential[0], ('P31212', 'THD1_SOLTU'))
        self.assertEqual(len(record.dr_unknown), 0)
        self.assertEqual(len(record.pdb_structs), 10)
        self.assertEqual(record.pdb_structs[0], "1E5X")
        self.assertEqual(record.pdb_structs[1], "1KL7")
        self.assertEqual(record.pdb_structs[2], "1P5J")
        self.assertEqual(record.pdb_structs[3], "1PWE")
        self.assertEqual(record.pdb_structs[4], "1PWH")
        self.assertEqual(record.pdb_structs[5], "1RSQ")
        self.assertEqual(record.pdb_structs[6], "1TDJ")
        self.assertEqual(record.pdb_structs[7], "1VB3")
        self.assertEqual(record.pdb_structs[8], "2C2B")
        self.assertEqual(record.pdb_structs[9], "2C2G")


if __name__ == "__main__":
    runner = unittest.TextTestRunner(verbosity = 2)
    unittest.main(testRunner=runner)
