#ifndef _RHEOLEF_FIELD_NONLINEAR_EXPR_TERMINAL_H
#define _RHEOLEF_FIELD_NONLINEAR_EXPR_TERMINAL_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// field_nonlinear_expr_terminal: terminals of nonn-linear expressions
// 1) class-function
//    1.1) base class for the class-function family
//    1.2) general function or class-function
//    1.3) normal to a surface
//    1.4) h_local
//    1.5) penalty
// 2) field and such
//    2.1) base class for the field family
//    2.2) field
//    2.3) grad of a field
//    2.4)  div of a field
//    2.5) curl of a field
//    2.6) jump of a field
// 3) convected field, as compose(uh,X) where X is a characteristic
//
#include "rheolef/field_nonlinear_expr.h"
#include "rheolef/field_evaluate.h"
#include "rheolef/characteristic.h"
#include "rheolef/test.h"

#include <boost/functional.hpp>

namespace rheolef {
// ---------------------------------------------------------------------------
// 1) class-function
// ---------------------------------------------------------------------------

template<class Function> class field_expr_terminal_function;

// wrapper, for backward compatibility:
template<class Function>
field_expr_terminal_function<Function>
field_function (const Function& f) {
  return field_expr_terminal_function<Function>(f);
}
// ---------------------------------------------------------------------------
// 1.1) base class for the class-function family
// ---------------------------------------------------------------------------
template<class T>
class field_expr_terminal_function_base_rep {
public:
// typedefs:

  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;

// allocators:

  field_expr_terminal_function_base_rep ();
  field_expr_terminal_function_base_rep (const field_expr_terminal_function_base_rep<T>&);

// accessors:

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  void initialize (const space_basic<float_type,memory_type>& Xh) const;
  void evaluate_initialize (const geo_element& K) const;
  const geo_element& get_side (const geo_element& K, const side_information_type& sid) const;

// data:
protected:
  mutable geo_basic<float_type,memory_type> _omega;
  mutable basis_on_pointset<float_type>     _piola_ops;
  mutable std::vector<size_type>            _dis_inod; // working data:
};
// ---------------------------------------------------------------------------
// 1.2) general function or class-function
// ---------------------------------------------------------------------------
template<class Function>
class field_expr_terminal_function_rep 
 : field_expr_terminal_function_base_rep<typename float_traits<typename boost::unary_traits<Function>::result_type>::type>
{
public:
// typedefs:

  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: how to deduce it ?
  typedef typename boost::unary_traits<Function>::function_type function_type;
  typedef typename boost::unary_traits<Function>::result_type   result_type;
  typedef typename boost::unary_traits<Function>::argument_type argument_type;
  typedef typename scalar_traits<result_type>::type             scalar_type;
  typedef typename  float_traits<result_type>::type             float_type;
  typedef result_type                                           value_type;
  typedef field_expr_terminal_function_base_rep<float_type>     base;

// alocators:

  field_expr_terminal_function_rep(const Function& f) : base(), _f(f) {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::valued_tag_traits<result_type>::value;

  space_constant::valued_type valued_tag() const {
      return space_constant::valued_tag_traits<result_type>::value; }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const {
    base::initialize (omega, quad);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
    base::initialize (Xh);
    return true;
  }
  void evaluate (const geo_element& K, std::vector<result_type>& value) const {
    base::evaluate_initialize (K);
    reference_element hat_K = K.variant();
    value.resize (base::_piola_ops.size(hat_K));
    size_type q = 0;
    for (typename std::vector<result_type>::iterator
	  iter = value.begin(),
	  last = value.end(); iter != last; ++iter, ++q) {
      point_basic<float_type> xq = piola_transformation (base::_omega, base::_piola_ops, K, base::_dis_inod, q);
      *iter = _f (xq);
    }
  }
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const {
    evaluate (base::get_side(L,sid), value);
  }
  template<class ValueType>
  void valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
  }
// data:
protected:
  function_type  _f;
};
template<class Function>
class field_expr_terminal_function : public smart_pointer<field_expr_terminal_function_rep<Function> >
{
public:
// typedefs:

  typedef field_expr_terminal_function_rep<Function> rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::argument_type           argument_type;
  typedef typename rep::value_type              value_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::float_type              float_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  explicit field_expr_terminal_function (const Function& f) : base(new_macro(rep(f))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  void evaluate (const geo_element& K, std::vector<result_type>& value) const {
  	base::data().evaluate (K, value); }
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const {
  	base::data().evaluate_on_side (L, sid, value); }
  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>(); }
};
// ---------------------------------------------------------------------------
// 1.3) normal to a surface
// ---------------------------------------------------------------------------
// the special class-function, used in nonlinear field expressions:
template<class T>
struct normal_pseudo_function : std::unary_function <point_basic<T>, point_basic<T> > {
  point_basic<T> operator() (const point_basic<T>&) const {
    fatal_macro ("special normal() class-function should not be directly evaluated");
    return point_basic<T>();
  }
};

template<class T>
class field_expr_terminal_function_rep<normal_pseudo_function<T> >
 : field_expr_terminal_function_base_rep<T> {
public:
// typedefs:

  typedef field_expr_terminal_function_base_rep<T>              base;
  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef typename boost::unary_traits<normal_pseudo_function<T> >::function_type function_type;
  typedef point_basic<T>                                        result_type;
  typedef point_basic<T>                                        argument_type;
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;
  typedef result_type                                           value_type;

// alocators:

  field_expr_terminal_function_rep (const function_type&);
  field_expr_terminal_function_rep (const field_expr_terminal_function_rep<normal_pseudo_function<T> >&);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::vector;

  space_constant::valued_type valued_tag() const { return space_constant::vector; }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  void evaluate         (const geo_element& K,                                   std::vector<point_basic<T> >& value) const;
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<point_basic<T> >& value) const;

  template<class ValueType>
  void valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
  }
};
// ---------------------------------------------------------------------------
// 1.4) h_local
// ---------------------------------------------------------------------------
// the special class-function, used in nonlinear field expressions:
template<class T>
struct h_local_pseudo_function : std::unary_function <point_basic<T>, T> {
  T operator() (const point_basic<T>&) const {
    fatal_macro ("special h_local() class-function should not be directly evaluated");
    return 0;
  }
};

template<class T>
class field_expr_terminal_function_rep<h_local_pseudo_function<T> >
 : field_expr_terminal_function_base_rep<T> {
public:
// typedefs:

  typedef field_expr_terminal_function_base_rep<T>              base;
  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef typename boost::unary_traits<h_local_pseudo_function<T> >::function_type function_type;
  typedef T                                                     result_type;
  typedef point_basic<T>                                        argument_type;
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;
  typedef result_type                                           value_type;

// alocators:

  field_expr_terminal_function_rep (const function_type&);
  field_expr_terminal_function_rep (const field_expr_terminal_function_rep<h_local_pseudo_function<T> >&);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::scalar;

  space_constant::valued_type valued_tag() const { return space_constant::scalar; }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  void evaluate         (const geo_element& K,                                   std::vector<result_type>& value) const;
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const;

  template<class ValueType>
  void valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
  }
};
// ---------------------------------------------------------------------------
// 1.5) penalty
// ---------------------------------------------------------------------------
// the special class-function, used in nonlinear field expressions:
template<class T>
struct penalty_pseudo_function : std::unary_function <point_basic<T>, T> {
  T operator() (const point_basic<T>&) const {
    fatal_macro ("special penalty() class-function should not be directly evaluated");
    return 0;
  }
};

template<class T>
class field_expr_terminal_function_rep<penalty_pseudo_function<T> >
 : field_expr_terminal_function_base_rep<T> {
public:
// typedefs:

  typedef field_expr_terminal_function_base_rep<T>              base;
  typedef geo_element::size_type                                size_type;
  typedef rheo_default_memory_model                             memory_type; // TODO: deduce it
  typedef typename boost::unary_traits<penalty_pseudo_function<T> >::function_type function_type;
  typedef T                                                     result_type;
  typedef point_basic<T>                                        argument_type;
  typedef T                                                     scalar_type;
  typedef T                                                     float_type;
  typedef result_type                                           value_type;

// alocators:

  field_expr_terminal_function_rep (const function_type&);
  field_expr_terminal_function_rep (const field_expr_terminal_function_rep<penalty_pseudo_function<T> >&);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::scalar;

  space_constant::valued_type valued_tag() const { return space_constant::scalar; }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  void evaluate         (const geo_element& K,                                   std::vector<result_type>& value) const;
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<result_type>& value) const;

  template<class ValueType>
  void valued_check() const {
    static const bool status = details::is_equal<ValueType,result_type>::value;
    check_macro (status, "unexpected result_type");
  }
protected:
// internal:
  T evaluate_measure  (const geo_element& K) const;
  void evaluate_internal (const geo_element& K, const geo_element& L, std::vector<result_type>& value) const;
};
// ---------------------------------------------------------------------------
// 2) field and such
// ---------------------------------------------------------------------------
// ---------------------------------------------------------------------------
// 2.1) base class for the field family
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_base_rep {
public:
// typedefs:

  typedef geo_element::size_type          size_type;

// alocators:

  field_expr_terminal_field_base_rep (const field_basic<T,M>& uh)
    : _uh(uh),   _omega(), _bops(), _use_dom2bgd(false), _use_bgd2dom(false), 
      _have_dg_on_sides(false), _tilde_L(), _dis_idof() {}

  template <class Expr>
  explicit field_expr_terminal_field_base_rep (const field_expr<Expr>& expr) 
    : _uh(expr), _omega(), _bops(), _use_dom2bgd(false), _use_bgd2dom(false), 
      _have_dg_on_sides(false), _tilde_L(), _dis_idof() {}

// accessors:

  const geo_basic<T,M>& get_geo() const { return _omega; }
  const geo_element& get_side (const geo_element& K, const side_information_type& sid) const;

  void initialize (const geo_basic<T,M>& omega, const quadrature<T>& quad) const;
  bool initialize (const space_basic<T,M>& Xh) const;
  void evaluate_initialize (const geo_element& K) const;
  void evaluate_on_side_initialize (const geo_element& L, const side_information_type& sid) const;

// data:
//protected:
  field_basic<T,M>                 _uh;
  mutable geo_basic<T,M>           _omega;
  mutable basis_on_pointset<T>     _bops;
  mutable bool                     _use_dom2bgd;
  mutable bool                     _use_bgd2dom;
  mutable bool                     _have_dg_on_sides;
  mutable reference_element        _tilde_L;
// working data:
  mutable std::vector<size_type>   _dis_idof;
};
// ---------------------------------------------------------------------------
// 2.2) field 
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_rep : public field_expr_terminal_field_base_rep<T,M> {
public:
// typedefs:

  typedef field_expr_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_terminal_field_rep (const field_basic<T,M>& uh) 
   : base(uh) {}

  template <class Expr>
  explicit field_expr_terminal_field_rep(const field_expr<Expr>& expr) 
    : base(expr) {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const { return base::_uh.valued_tag(); }

  const geo_basic<T,M>& get_geo() const { return base::get_geo(); }
  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  void evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const;
  template<class ValueType>
  void valued_check() const {
    space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
    check_macro ((base::_uh.valued_tag() == valued_tag) ||
                 (base::_uh.valued_tag() == space_constant::unsymmetric_tensor &&
                  valued_tag             == space_constant::tensor),
           "unexpected "<< base::_uh.valued()
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
  }
};
template<class T, class M>
class field_expr_terminal_field : public smart_pointer<field_expr_terminal_field_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_terminal_field_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_terminal_field(const field_basic<T,M>& uh)
    : base(new_macro(rep(uh))) {} 

  template <class Expr>
  explicit field_expr_terminal_field(const field_expr<Expr>& expr) 
    : base(new_macro(rep(expr))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  const geo_basic<T,M>& get_geo() const { return base::data().get_geo(); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>();
  }
  template<class ValueType>
  void evaluate (const geo_element& K, std::vector<ValueType>& value) const {
  	base::data().evaluate(K,value);
  } 
  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const {
  	base::data().evaluate_on_side (L, sid, value); }
};
// ---------------------------------------------------------------------------
// 2.3) grad of a field
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_grad_rep : public field_expr_terminal_field_base_rep<T,M> {
public:
// typedefs:

  typedef field_expr_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_terminal_field_grad_rep (const field_basic<T,M>& expr, const details::grad_option_type& opt)
    : base(expr), 
      _opt(opt), 
      _u_test(base::_uh.get_space())
    {}

  template <class Expr>
  explicit field_expr_terminal_field_grad_rep (const field_expr<Expr>& expr, const details::grad_option_type& opt) 
    : base(expr), 
      _opt(opt), 
      _u_test(base::_uh.get_space())
    {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const;

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  void evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  void valued_check() const { _u_test.template grad_valued_check<ValueType>(); }
protected:
// data:
  details::grad_option_type          _opt;
  test_basic<T,M,details::vf_tag_01> _u_test;
};

template<class T, class M>
class field_expr_terminal_field_grad : public smart_pointer<field_expr_terminal_field_grad_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_terminal_field_grad_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_terminal_field_grad (const field_basic<T,M>& uh, const details::grad_option_type& opt = details::grad_option_type())
    : base(new_macro(rep(uh,opt))) {} 

  template <class Expr>
  explicit field_expr_terminal_field_grad (const field_expr<Expr>& expr, const details::grad_option_type& opt = details::grad_option_type()) 
    : base(new_macro(rep(expr,opt))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  template<class Result>
  void evaluate (const geo_element& K, std::vector<Result>& value) const {
  	base::data().evaluate(K,value); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>();
  }
};
// ---------------------------------------------------------------------------
// 2.4) div of a field
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_div_rep : public field_expr_terminal_field_base_rep<T,M> {
public:
// typedefs:


  typedef field_expr_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
#ifdef TODO
  typedef undeterminated_basic<T>         result_type; // div(tensor) -> vector : TODO
#endif // TODO
  typedef T		                  result_type; // div(vector) -> scalar
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_terminal_field_div_rep (const field_basic<T,M>& expr, const details::grad_option_type& opt)
    : base(expr),
      _opt(opt), 
      _u_test(base::_uh.get_space())
    {}

  template <class Expr>
  explicit field_expr_terminal_field_div_rep (const field_expr<Expr>& expr, const details::grad_option_type& opt) 
    : base(expr),
      _opt(opt), 
      _u_test(base::_uh.get_space())
    {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const;

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  void evaluate (const geo_element& K, std::vector<ValueType>& value) const;

  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const {
    base::evaluate_on_side_initialize (L, sid);
    evaluate (L, value);
  }
  template<class ValueType>
  void valued_check() const { _u_test.template div_valued_check<ValueType>(); }
// data:
  details::grad_option_type          _opt;
  test_basic<T,M,details::vf_tag_01> _u_test;
};

template<class T, class M>
class field_expr_terminal_field_div : public smart_pointer<field_expr_terminal_field_div_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_terminal_field_div_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_terminal_field_div (const field_basic<T,M>& uh, const details::grad_option_type& opt = details::grad_option_type())
    : base(new_macro(rep(uh,opt))) {} 

  template <class Expr>
  explicit field_expr_terminal_field_div (const field_expr<Expr>& expr, const details::grad_option_type& opt = details::grad_option_type()) 
    : base(new_macro(rep(expr,opt))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  template<class Result>
  void evaluate (const geo_element& K, std::vector<Result>& value) const {
  	base::data().evaluate(K,value); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  void evaluate_on_side (const geo_element& L, const side_information_type& sid, std::vector<ValueType>& value) const {
  	base::data().evaluate_on_side (L, sid, value);
  }
  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>();
  }
};
// ---------------------------------------------------------------------------
// 2.5) curl of a field
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_curl_rep : public field_expr_terminal_field_base_rep<T,M> {
public:
// typedefs:

  typedef field_expr_terminal_field_base_rep<T,M> base;
  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_terminal_field_curl_rep (const field_basic<T,M>& expr, const details::grad_option_type& opt)
   : base(expr),
     _opt(opt), 
     _u_test(base::_uh.get_space())
    {}

  template <class Expr>
  explicit field_expr_terminal_field_curl_rep (const field_expr<Expr>& expr, const details::grad_option_type& opt) 
   : base(expr),
     _opt(opt), 
     _u_test(base::_uh.get_space())
    {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const;

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  void evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  void valued_check() const { _u_test.template curl_valued_check<ValueType>(); }
// data:
  details::grad_option_type          _opt;
  test_basic<T,M,details::vf_tag_01> _u_test;
};

template<class T, class M>
class field_expr_terminal_field_curl : public smart_pointer<field_expr_terminal_field_curl_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_terminal_field_curl_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_terminal_field_curl (const field_basic<T,M>& uh, const details::grad_option_type& opt = details::grad_option_type())
    : base(new_macro(rep(uh,opt))) {} 

  template <class Expr>
  explicit field_expr_terminal_field_curl (const field_expr<Expr>& expr, const details::grad_option_type& opt = details::grad_option_type()) 
    : base(new_macro(rep(expr,opt))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  template<class Result>
  void evaluate (const geo_element& K, std::vector<Result>& value) const {
  	base::data().evaluate(K,value); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>();
  }
};
// ----------------------------------------------------------------------------
// 2.6) jump of a field
// ----------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_dg_rep {
public:
// typedefs:

  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_terminal_field_dg_rep (const field_basic<T,M>& uh, const float_type& c0, const float_type& c1)
   : _expr0(uh), _expr1(uh),
     _c0(c0), _c1(c1)
   {}

  template <class Expr>
  explicit field_expr_terminal_field_dg_rep(const field_expr<Expr>& expr, const float_type& c0, const float_type& c1)
   : _expr0(expr), _expr1(expr),
     _c0(c0), _c1(c1)
   {}

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const { return _expr0.valued_tag(); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& quad) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;
  template<class ValueType>
  void evaluate (const geo_element& K, std::vector<ValueType>& value) const;
  template<class ValueType>
  void valued_check() const {
    space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
    check_macro (_expr0.valued_tag() == valued_tag,
 	              "unexpected "<<  space_constant::valued_name(_expr0.valued_tag())
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
  }
// data:
protected:
  field_expr_terminal_field<T,M> _expr0, _expr1;
  float_type                     _c0,    _c1;
};

template<class T, class M>
class field_expr_terminal_field_dg : public smart_pointer<field_expr_terminal_field_dg_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_terminal_field_dg_rep<T,M> rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_terminal_field_dg (const field_basic<T,M>& uh, const float_type& c0, const float_type& c1)
    : base(new_macro(rep(uh,c0,c1))) {} 

  template <class Expr>
  explicit field_expr_terminal_field_dg(const field_expr<Expr>& expr, const float_type& c0, const float_type& c1)
    : base(new_macro(rep(expr,c0,c1))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>();
  }
  template<class Result>
  void evaluate (const geo_element& K, std::vector<Result>& value) const {
  	base::data().evaluate(K,value);
  } 
};
// ---------------------------------------------------------------------------
// 3) convected field, as compose(uh,X) where X is a characteristic
// ---------------------------------------------------------------------------
template<class T, class M>
class field_expr_terminal_field_o_characteristic_rep {
public:
// typedefs:

  typedef geo_element::size_type          size_type;
  typedef M                               memory_type;
  typedef undeterminated_basic<T>         result_type;
#ifdef TO_CLEAN
  typedef T                               result_type;
#endif // TO_CLEAN
  typedef result_type                     value_type;
  typedef T                               scalar_type;
  typedef typename float_traits<T>::type  float_type;

// alocators:

  field_expr_terminal_field_o_characteristic_rep (const field_o_characteristic<T,M>& uoX);
  field_expr_terminal_field_o_characteristic_rep (const field_basic<T,M>& uh, const characteristic_basic<T,M>& X);

// accessors:

  static const space_constant::valued_type valued_hint = space_constant::last_valued;

  space_constant::valued_type valued_tag() const { return _uh.valued_tag(); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const;
  bool initialize (const space_basic<float_type,memory_type>& Xh) const;

  template<class Result>
  void evaluate (const geo_element& K, std::vector<Result>& value) const;

  template<class ValueType>
  void valued_check() const {
    space_constant::valued_type valued_tag = space_constant::valued_tag_traits<ValueType>::value;
    check_macro (_uh.valued_tag() == valued_tag, "unexpected "<<_uh.valued()
        << "-valued field while a " << space_constant::valued_name(valued_tag)
        << "-valued one is expected in expression");
  }

// internal:
  template<class Result>
  void _check () const;

// data:
  field_basic<T,M>                            _uh;
  characteristic_basic<T,M>                   _X;
  mutable bool                                _is_initialized;
  mutable quadrature<T>                       _quad;
  mutable basis_on_pointset<T>                _bops;
  mutable array<T,M>                          _uq;
  mutable typename array<T,M>::const_iterator _uq_K;
  mutable size_type                           _start_q;
};

template<class T, class M>
class field_expr_terminal_field_o_characteristic : 
  public smart_pointer<field_expr_terminal_field_o_characteristic_rep<T,M> >
{
public:
// typedefs:

  typedef field_expr_terminal_field_o_characteristic_rep<T,M>    rep;
  typedef smart_pointer<rep>                    base;
  typedef typename rep::size_type               size_type;
  typedef typename rep::memory_type             memory_type;
  typedef typename rep::result_type             result_type;
  typedef typename rep::float_type              float_type;
  typedef typename rep::scalar_type             scalar_type;
  typedef typename rep::value_type              value_type;
  static const space_constant::valued_type valued_hint = rep::valued_hint;

// alocators:

  field_expr_terminal_field_o_characteristic (const field_o_characteristic<T,M>& uoX)
    : base(new_macro(rep(uoX))) {} 

  field_expr_terminal_field_o_characteristic (const field_basic<T,M>& uh, const characteristic_basic<T,M>& X)
    : base(new_macro(rep(uh,X))) {} 

// accessors:

  space_constant::valued_type valued_tag() const { return base::data().valued_tag(); }

  void initialize (const geo_basic<float_type,memory_type>& omega, const quadrature<float_type>& hat_x) const {
  	base::data().initialize(omega,hat_x);
  }
  bool initialize (const space_basic<float_type,memory_type>& Xh) const {
  	return base::data().initialize(Xh);
  }
  template<class Result>
  void evaluate (const geo_element& K, std::vector<Result>& value) const {
  	base::data().evaluate(K,value); }

  template<class ValueType>
  void valued_check() const {
  	base::data().valued_check<ValueType>();
  }
};

} // namespace rheolef
#endif // _RHEOLEF_FIELD_NONLINEAR_EXPR_TERMINAL_H
