/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/binomial/cdf' );
import Binomial = require( './../../../../../base/dists/binomial/ctor' );
import entropy = require( './../../../../../base/dists/binomial/entropy' );
import kurtosis = require( './../../../../../base/dists/binomial/kurtosis' );
import logpmf = require( './../../../../../base/dists/binomial/logpmf' );
import mean = require( './../../../../../base/dists/binomial/mean' );
import median = require( './../../../../../base/dists/binomial/median' );
import mgf = require( './../../../../../base/dists/binomial/mgf' );
import mode = require( './../../../../../base/dists/binomial/mode' );
import pmf = require( './../../../../../base/dists/binomial/pmf' );
import quantile = require( './../../../../../base/dists/binomial/quantile' );
import skewness = require( './../../../../../base/dists/binomial/skewness' );
import stdev = require( './../../../../../base/dists/binomial/stdev' );
import variance = require( './../../../../../base/dists/binomial/variance' );

/**
* Interface describing the `binomial` namespace.
*/
interface Namespace {
	/**
	* Binomial distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param n - number of trials
	* @param p - success probability
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 3.0, 20, 0.2 );
	* // returns ~0.411
	*
	* y = ns.cdf( 21.0, 20, 0.2 );
	* // returns 1.0
	*
	* y = ns.cdf( 5.0, 10, 0.4 );
	* // returns ~0.834
	*
	* y = ns.cdf( 0.0, 10, 0.4 );
	* // returns ~0.06
	*
	* var mycdf = ns.cdf.factory( 10, 0.5 );
	*
	* y = mycdf( 3.0 );
	* // returns ~0.172
	*
	* y = mycdf( 1.0 );
	* // returns ~0.011
	*/
	cdf: typeof cdf;

	/**
	* Binomial Distribution.
	*/
	Binomial: typeof Binomial;

	/**
	* Returns the entropy of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 100, 0.1 );
	* // returns ~2.511
	*
	* @example
	* var v = ns.entropy( 20, 0.5 );
	* // returns ~2.223
	*
	* @example
	* var v = ns.entropy( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 20, NaN );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the kurtosis of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 100, 0.1 );
	* // returns ~0.051
	*
	* @example
	* var v = ns.kurtosis( 20, 0.5 );
	* // returns ~-0.1
	*
	* @example
	* var v = ns.kurtosis( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 20, NaN );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Binomial distribution natural logarithm of the probability mass function (logPMF).
	*
	* @param x - input value
	* @param n - number of trials
	* @param p - success probability
	* @returns evaluated logPMF
	*
	* @example
	* var y = ns.logpmf( 3.0, 20, 0.2 );
	* // returns ~-1.583
	*
	* y = ns.logpmf( 21.0, 20, 0.2 );
	* // returns -Infinity
	*
	* y = ns.logpmf( 5.0, 10, 0.4 );
	* // returns ~-1.606
	*
	* y = ns.logpmf( 0.0, 10, 0.4 );
	* // returns ~-5.108
	*
	* var mylogpmf = ns.logpmf.factory( 10, 0.5 );
	*
	* y = mylogpmf( 3.0 );
	* // returns ~-2.146
	*
	* y = mylogpmf( 5.0 );
	* // returns ~-1.402
	*/
	logpmf: typeof logpmf;

	/**
	* Returns the expected value of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 100, 0.1 );
	* // returns 10.0
	*
	* @example
	* var v = ns.mean( 20, 0.5 );
	* // returns 10.0
	*
	* @example
	* var v = ns.mean( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 20, NaN );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns median
	*
	* @example
	* var v = ns.median( 100, 0.1 );
	* // returns 10
	*
	* @example
	* var v = ns.median( 20, 0.5 );
	* // returns 10
	*
	* @example
	* var v = ns.median( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 20, NaN );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Binomial distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param n - number of trials
	* @param p - success probability
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, 20, 0.2 );
	* // returns ~11.471
	*
	* y = ns.mgf( 5.0, 20, 0.2 );
	* // returns ~4.798e29
	*
	* y = ns.mgf( 0.9, 10, 0.4 )
	* // returns ~99.338
	*
	* var mymgf = ns.mgf.factory( 10, 0.5 );
	*
	* y = mymgf( 0.3 );
	* // returns ~5.013
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns mode
	*
	* @example
	* var v = ns.mode( 100, 0.1 );
	* // returns 10
	*
	* @example
	* var v = ns.mode( 20, 0.5 );
	* // returns 10
	*
	* @example
	* var v = ns.mode( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 20, NaN );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Binomial distribution probability mass function (PMF).
	*
	* @param x - input value
	* @param n - number of trials
	* @param p - success probability
	* @returns evaluated PMF
	*
	* @example
	* var y = ns.pmf( 3.0, 20, 0.2 );
	* // returns ~0.205
	*
	* y = ns.pmf( 21.0, 20, 0.2 );
	* // returns 0.0
	*
	* y = ns.pmf( 5.0, 10, 0.4 );
	* // returns ~0.201
	*
	* y = ns.pmf( 0.0, 10, 0.4 );
	* // returns ~0.06
	*
	* var mypmf = ns.pmf.factory( 10, 0.5 );
	*
	* y = mypmf( 3.0 );
	* // returns ~0.117
	*
	* y = mypmf( 5.0 );
	* // returns ~0.246
	*/
	pmf: typeof pmf;

	/**
	* Binomial distribution quantile function.
	*
	* @param p - input value
	* @param n - number of trials
	* @param p - success probability
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.4, 20, 0.2 );
	* // returns 2
	*
	* y = ns.quantile( 0.8, 20, 0.2 );
	* // returns 5
	*
	* y = ns.quantile( 0.5, 10, 0.4 );
	* // returns 4
	*
	* var myQuantile = ns.quantile.factory( 10, 0.5 );
	*
	* y = myQuantile( 0.1 );
	* // returns 3
	*
	* y = myQuantile( 0.9 );
	* // returns 7
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 100, 0.1 );
	* // returns ~0.267
	*
	* @example
	* var v = ns.skewness( 20, 0.5 );
	* // returns 0.0
	*
	* @example
	* var v = ns.skewness( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 20, NaN );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 100, 0.1 );
	* // returns 3.0
	*
	* @example
	* var v = ns.stdev( 20, 0.5 );
	* // returns ~2.236
	*
	* @example
	* var v = ns.stdev( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 20, NaN );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a binomial distribution.
	*
	* ## Notes
	*
	* -   If provided a number of trials `n` which is not a nonnegative integer, the function returns `NaN`.
	* -   If `p < 0` or `p > 1`, the function returns `NaN`.
	*
	* @param n - number of trials
	* @param p - success probability
	* @returns variance
	*
	* @example
	* var v = ns.variance( 100, 0.1 );
	* // returns 9.0
	*
	* @example
	* var v = ns.variance( 20, 0.5 );
	* // returns 5.0
	*
	* @example
	* var v = ns.variance( 10.3, 0.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 20, 1.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 20, NaN );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Binomial distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
